{ interface }
const
CharTimeoutErr                  = 1;
CharDataErr                     = 2;
TimeoutIRBFIFONotEmpty          = 3;
TimeoutICDFIFONotFull           = 4;
TimeoutICDFIFONotHalfFull       = 5;
TimeoutQRBFIFONotEmpty          = 6;
TimeoutQCDFIFONotFull           = 7;
TimeoutQCDFIFONotHalfFull       = 8;
TimeoutQRBFIFOHalfFull          = 9;
TimeOutIRBFIFOHalfFull          = 10;
LastError                       = 11;
type

errrec =
record
  errnum : integer;
  errstr :string[50];
end;

const
errorrecord : array[1 .. LastError] of errrec =
  (
  (errnum : CharTimeoutErr;            errstr : 'Time out waiting for Character'),
  (errnum : CharDataErr;               errstr : 'Character Data Error'),
  (errnum : TimeoutIRBFIFONotEmpty;    errstr : 'Time out waiting for IRF FIFO not empty'),
  (errnum : TimeoutICDFIFONotFull;     errstr : 'Time out waiting for ICD FIFO not full'),
  (errnum : TimeoutICDFIFONotHalfFull; errstr : 'Time out waiting for ICD FIFO not half full'),
  (errnum : TimeoutQRBFIFONotEmpty;    errstr : 'Time out waiting for QRF FIFO not empty'),
  (errnum : TimeoutQCDFIFONotFull;     errstr : 'Time out waiting for QCD FIFO not full'),
  (errnum : TimeoutQCDFIFONotHalfFull; errstr : 'Time out waiting for QCD FIFO not half full'),
  (errnum : TimeoutQRBFIFOHalfFull;    errstr : 'Time out waiting for QRB FIFO not empty'),
  (errnum : TimeoutIRBFIFOHalfFull;    errstr : 'Time out waiting for IRB FIFO not empty'),
  (errnum : LastError;                 errstr : '')
  );

{$IFNDEF WINDOWS}
procedure DisableInterrupts;
begin
  inline($FA);                                 { CLI }
end;

procedure EnableInterrupts;
begin
  inline($FB);                                 { STI }
end;
{$ENDIF NOT WINDOWS}

const
CharTimeOutSeconds = 1;
PollTimeOutSeconds = 1;
BusTimeOut = 500000*PollTimeOutSeconds;
BitsPerChar = 10;
LoopsPermS = 400;
PollChars = 8;
AxisOffset = 10;
ProcResetBit : word = $8000;
LongFlag : word = $2000;
WaitTokenFlag : word = $4000;
DoubleFlag : word = $6000;
WriteFlag : word = $8000;
BlockFlag : word = $C000;
ROMSize1 = 1024;
ROMSize2 = 2048;
SignatureOffset = -1;
Signature = $CAFE;
DefaultPC104BasePort0 = $220;
DefaultPC104BasePort1 = $230;
DefaultPC104BasePort2 = $240;
DefaultPC104BasePort3 = $250;

DefaultPC104BasePort = DefaultPC104BasePort0;

DefaultLBPBasePort = $80;

IFIFOPortOffset = 0;
QFIFOPortOffset = 4;
ROMAddOffset =  8;
StatusRegOffset = 8;
ROMDataOffset  = 12;

type
ProtocolTypes = (Bus,Hex,LBP,Invalid);
Cardtypes =  (PC104,PCI,SevenI60,ThreeC20,SevenI43,EightC20);
ParamArray = array[1..1024] of word;

var
Axis : byte;
ProtocolType : ProtocolTypes;
CardType : cardtypes;
PC104Port,PCIPort : word;
IFIFOPort : word;
QFIFOPort : word;
ROMAddPort : word;
StatusPort : word;
ROMDataPort : word;
PollTimeOut : longint;

procedure Error(err : integer); forward;

procedure BumOut(es : string);
begin
  writeln(es);
end;

{$I parsep}
{$I hread.pas}
{$I hprint.pas}

{$I sercmds4}

{$I DataRec}
{$I Userial}

{$IFNDEF WINDOWS}

function KeyPressed : boolean;
var flag : boolean;
begin
  asm
    mov ah,01
    int $16
    mov flag,false
    jz @nokey
    mov flag,true
    @nokey:
  end;
  KeyPressed := flag;
end;
{$ENDIF}


procedure GetOurEnv;
var
retcode : integer;
portstr,baudstr,baudmulstr,protostr,pc104Addstr,scomport : string;
comnumber,ourcomport : longint;
begin
  PollTimeout := BusTimeOut;

  portstr := UpString(GetEnv('COMPORT'));

  baudstr := GetEnv('BAUDRATE');
  baudmulstr := GetEnv('BAUDRATEMUL');

  protostr := GetEnv('PROTOCOL');
  pc104Addstr := GetEnv('PC104ADD');

  ProtocolType := invalid;
  protostr := upstring(protostr);
  if length(protostr) = 0 then
  begin
    Writeln('Protocol not in Environment, assuming Hex');
    ProtocolType := Hex;
  end
  else
  begin
    if protostr = 'BUS' then ProtocolType := Bus;
    if protostr = 'LBP' then ProtocolType := LBP;
    if protostr = 'HEX' then ProtocolType := Hex;
  end;
  case ProtocolType of
    Bus:
    begin
{$IFNDEF WINDOWS}
      Pc104Port := DefaultPC104BasePort;
      if length(pc104addstr) <> 0 then
      if HexWordRead(pc104addstr,Pc104Port) then
      begin
        if (Pc104Port < $220) or (Pc104Port >$250) or (Pc104Port and $000F <> 0) then Bumout('Invalid PC104 address');
      end else Bumout('Can''t parse PC/104 address');
{$ELSE}
      Bumout('Protocol not supported');
{$ENDIF}
    end; {bus}
    Hex,LBP: { serial }
    begin
      ComPortHardwareType := STDComPort; { default }

      if length(portstr) = 0 then
      begin
        Writeln('Comport not in Environment, assuming COM1');
        portstr := 'COM1';
      end;
      { now parse comport }
{$IFDEF WINDOWS}
      if copy(portstr,1,3) = 'COM' then
      begin
        val(copy(portstr,4,length(portstr)),comnumber,retcode);
        if retcode = 0 then scomport := portstr else Bumout('Undecipherable Windows COM port - bail out!');
      end
      else Bumout('Undecipherable Windows COM port - bail out!');
{$ELSE}
      { check for hex address }
      if (HexLongRead(portstr,ourcomport)) then
      begin
        if (ourcomport <$100) or (ourcomport >$FFF8) then Bumout ('Out of range COM port address - bail out!');
        if (ourcomport and 7) <> 0 then Bumout('Illegal COM port address - bail out!');
        ComPortHardwareType := STDComPort{HEXComPort};
      end;

      { check for mesa }
      if copy(portstr,1,4) = 'MCOM' then
      begin
        val(copy(portstr,5,length(portstr)),comnumber,retcode);
        if retcode = 0 then
        begin
          ourcomport := comnumber-1 ;
          ComPortHardwareType := MESAComPort;
        end else  Bumout('Undecipherable MCOM port - bail out!');
      end;

      if ComPortHardwareType = STDComPort then
      begin
        ourcomport := 0;
        if portstr = 'COM1' then ourcomport := COM1;
        if portstr = 'COM2' then ourcomport := COM2;
        if portstr = 'COM3' then ourcomport := COM3;
        if portstr = 'COM4' then ourcomport := COM4;
        if ourcomport = 0 then Bumout('Undecipherable COM port - bail out!')
      end; {comporttype }
{$ENDIF}

      if length(baudmulstr) = 0 then
      begin
        Writeln('BaudrateMul not in Environment, assuming 1');
        TheBaudRateMul := 1;
      end else
      begin
        val(baudmulstr,TheBaudRateMul,retcode);
        if (TheBaudRateMul <1) or (TheBaudRateMul >64) or (retcode <> 0) then
        begin
          Bumout('Illegal Baud rate Multiplier - bail out!');
        end;
      end; { baudmulstr }

      if length(baudstr) = 0 then
      begin
        Writeln('Baudrate not in Environment, assuming 115200');
        TheBaudRate := 115200;
      end else
      begin
        val(baudstr,TheBaudRate,retcode);

        case ComPortHardwareType of
          STDComPort:
          begin
            if (TheBaudRate < (300 * TheBaudrateMul)) or (TheBaudRate > (115200 * TheBaudRateMul)) or (retcode <> 0) then
            begin
              Bumout('Illegal Baud rate - bail out!');
            end;
          end;
{$IFNDEF WINDOWS}
          MESAComPort:
          begin
            if not MesaBaudRateValid(TheBaudRate,TheBaudRateMul) then Bumout('Illegal Baud rate - bail out!');
          end;
{$ENDIF}
        end;
      end;
{$IFDEF WINDOWS}
      CharTimeout := CharTimeOutSeconds*1000;
      TheComPort := scomport;
{$ELSE}
      CharTimeout := CharTimeOutSeconds*1000*LoopsPermS;
      TheComPort := word(ourcomport);
{$ENDIF}
      PollTimeOut := trunc((PollTimeOutSeconds*TheBaudRate*TheBaudRateMul)/(BitsPerChar*PollChars));
    end;  { end serial }
    Invalid: Bumout('Invalid Protocol');
  end;
end;

{ motlow4 stuff}
function SerReadFifoStatus: word;
var
data : word;
rstring,sstring : string;
retchar : char;
begin
  sstring := SerReadFifoStatusCom + CR;
  SerSendString(sstring);
  SerRecvString(4,rstring);  { get 4 char hex string }
  if not SerRecvChar(retchar) then SerError := true; { get CR }
  HexWordRead(rstring,data);
  SerReadFifoStatus := data;
end;

{$IFDEF WINDOWS}
  {$IFDEF FIFO}
     var portw :  array [0..65535] of word;
  {$ENDIF FIFO}
{$ELSE}
  {$IFDEF FIFO}
  {$ENDIF FIFO}
{$ENDIF}

{$I serhost4.pas}
{$I parhost4.pas}

{$I uLBPhost.pas}

{$IFDEF FIFO}
procedure WriteRom(add : word;data : word);
begin
   case ProtocolType of
     Bus:  BusWriteRom(add,data);
    { Hex: HexWriteRom(add,data); }
     LBP:  LBPWriteRom(add,data);
   end;
end;

function ReadRom(add : word): word;
begin
   case ProtocolType of
     Bus:  ReadRom := BusReadRom(add);
    { Hex: ReadRom(add) := HexReadRom(add);}
     LBP:  ReadRom := LBPReadRom(add);
   end;
end;
{$ENDIF FIFO}

{IFNDEF WINDOWS}

procedure WriteWord(add : word;data : word);
begin
   case ProtocolType of
     Hex: SerWritePicWord((add*2)+2048,data);
     LBP: LBPWriteWord(add,data);
   end;
end;

function ReadWord(add : word): word;
begin
   case ProtocolType of
     Hex: ReadWord := SerReadPicWord((add*2)+2048);  { for parameter compatibility }
     LBP: ReadWord := LBPReadWord(add);
   end;
end;

{$IFDEF EEPROM}
procedure WriteEEPROM(add : word;data : byte);
begin
   case ProtocolType of
     Hex: SerWriteEEPROM(add,data);
     LBP: LBPWriteEEPROM(add,data);
   end;
end;

function ReadEEPROM(add : word): byte;
begin
   case ProtocolType of
     Hex: ReadEEPROM := SerReadEEPROM(add);
     LBP: ReadEEPROM := LBPReadEEPROM(add);
   end;
end;

procedure WriteEEPROMWord(add : word;data : word);
begin
   case ProtocolType of
     Hex: SerWriteEEPROMWord(add,data);
     LBP: LBPWriteEEPROMWord(add,data);
   end;
end;

function ReadEEPROMWord(add : word): word;
begin
   case ProtocolType of
     Hex: ReadEEPROMWord := SerReadEEPROMWord(add);
     LBP: ReadEEPROMWord := LBPReadEEPROMWord(add);
   end;
end;
{$ENDIF EEPROM}

procedure WriteLong(add : word;data : longint);
begin
   case ProtocolType of
     Hex:
       begin
         SerWritePicWord((add*2)+2048,LongIntRec(data).LowWord);
         SerWritePicWord((add*2)+2048+2,LongIntRec(data).HighWord);
       end;
     LBP: LBPWriteLong(add,data);
   end;
end;

procedure WriteDouble(add : word;data : comp);
begin
   case ProtocolType of
     Hex:
       begin
         SerWritePicWord((add*2)+2048,DoubleIntRec(data).Word0);
         SerWritePicWord((add*2)+2048+2,DoubleIntRec(data).Word1);
         SerWritePicWord((add*2)+2048+4,DoubleIntRec(data).Word2);
         SerWritePicWord((add*2)+2048+6,DoubleIntRec(data).Word3);
       end;
     LBP: LBPWriteDouble(add,data);
   end;
end;

function ReadLong(add : word): longint;
var rwl,rwh : word;
rl : longint;
begin
   case ProtocolType of
     Hex:
       begin
         rwl := SerReadPicWord((add*2)+2048);
         rwh := SerReadPicWord((add*2)+2048+2);
         LongIntRec(rl).LowWord := rwl;
         LongIntRec(rl).HighWord := rwh;
         ReadLong := rl;
       end;
     LBP: ReadLong := LBPReadLong(add);
   end;
end;

function ReadDouble(add : word): comp;
var rw0,rw1,rw2,rw3 : word;
rd : comp;
begin
   case ProtocolType of
     Hex:
       begin
         rw0 := SerReadPicWord((add*2)+2048);
         rw1 := SerReadPicWord((add*2)+2048+2);
         rw2 := SerReadPicWord((add*2)+2048+4);
         rw3 := SerReadPicWord((add*2)+2048+6);
         DoubleIntRec(rd).Word0 := rw0;
         DoubleIntRec(rd).Word1 := rw1;
         DoubleIntRec(rd).Word2 := rw2;
         DoubleIntRec(rd).Word3 := rw3;
         DoubleIntRec(rd).Word0 := rw0;
         ReadDouble := rd;
       end;
     LBP: ReadDouble := LBPReadDouble(add);
   end;
end;
{ENDIF }

{$IFDEF FIFO}
procedure SoftDmcResetOn;
begin
   case ProtocolType of
     Bus:  BusSoftDMCResetOn;
     Hex: ;
     LBP:  LBPSoftDMCResetOn;
   end;
end;

procedure SoftDmcResetOff;
begin
   case ProtocolType of
     Bus:  BusSoftDMCResetOff;
     Hex: { WHAT ABOUT 3C20  AND 7I60};
     LBP:  LBPSoftDMCResetOff;
   end;
end;

function ReadFIFOStatus : word;
begin
   case ProtocolType of
     Bus: ReadFIFOStatus := BusReadFifoStatus;
     Hex: ReadFIFOStatus := SerReadFifoStatus;
     LBP: ReadFIFOStatus := LBPReadFifoStatus;
   end;
end;

procedure WaitForIRBFIFONotEmpty;
var loop : longint;
begin
  loop := 0;
  repeat
    loop := loop +1;
  until ((ReadFIFOStatus and IRE = 0) or (loop > PollTimeOut));
  if loop > PollTimeOut then Error(TimeOutIRBFIFONotEmpty);
end;

procedure WaitForICDFIFONotFull;
var loop : longint;
begin
  loop := 0;
  repeat
    loop := loop +1;
  until ((ReadFIFOStatus and IFF = 0) or (loop > PollTimeOut));
  if loop > PollTimeOut then Error(TimeOutICDFIFONotFull);
end;

procedure WaitForICDFIFONotHalfFull;
var loop : longint;
begin
  loop := 0;
  repeat
    loop := loop +1;
  until ((ReadFIFOStatus and IFH = 0)  or (loop > PollTimeOut));
  if loop > PollTimeOut then Error(TimeOutICDFIFONotHalfFull);
end;

procedure WaitForQRBFIFONotEmpty;
var loop : longint;
begin
  loop := 0;
  repeat
    loop := loop +1;
  until ((ReadFIFOStatus and QRE = 0)  or (loop > PollTimeOut));
  if loop > PollTimeOut then Error(TimeOutQRBFIFONotEmpty);
end;

procedure WaitForQCDFIFONotFull;
var loop : longint;
begin
  loop := 0;
  repeat
    loop := loop +1;
  until ((ReadFIFOStatus and QFF = 0)  or (loop > PollTimeOut));
  if loop > PollTimeOut then Error(TimeOutQCDFIFONotFull);
end;

procedure WaitForQCDFIFONotHalfFull;
var loop : longint;
begin
  loop := 0;
  repeat
    loop := loop +1;
  until ((ReadFIFOStatus and QFH = 0)  or (loop > PollTimeOut));
  if loop > PollTimeOut then Error(TimeOutQCDFIFONotHalfFull);
end;

procedure WaitForQRBFIFOHalfFull;
var loop : longint;
begin
  loop := 0;
  repeat
    loop := loop +1;
  until ((ReadFIFOStatus and QRH <> 0)  or (loop > PollTimeOut));
  if loop > PollTimeOut then Error(TimeOutQRBFIFOHalfFull);
end;

procedure WaitForIRBFIFOHalfFull;
var loop : longint;
begin
  loop := 0;
  repeat
    loop := loop +1;
  until ((ReadFIFOStatus and IRH <> 0)  or (loop > PollTimeOut));
  if loop > PollTimeOut then Error(TimeOutIRBFIFOHalfFull);
end;

procedure WriteCommand(theaxis: byte;theparm: word);
var command : word;
{ 16 bit parameter write }
begin
  WaitForICDFIFONotHalfFull;
  command := (theparm or (theaxis shl AxisOffset));
   case ProtocolType of
     Bus: BusWriteCommand(command);
     Hex: SerWriteCommand(command);
     LBP: LBPWriteCommand(command);
   end;
end;

procedure WriteCommandQ(theaxis: byte;theparm: word);
var command : word;
{ 16 bit parameter write }
begin
  WaitForQCDFIFONotHalfFull;
  command := (theparm or (theaxis shl AxisOffset));
   case ProtocolType of
     Bus: BusWriteCommandQ(command);
     Hex: SerWriteCommandQ(command);
     LBP: LBPWriteCommandQ(command);
   end;
end;

procedure WriteParamWord(theaxis: byte;theparm: word;thedata : word);
var command : word;
{ 16 bit parameter write }
begin
  WaitForICDFIFONotHalfFull;
  command := (theparm or WriteFlag or (theaxis shl AxisOffset));
   case ProtocolType of
     Bus: BusWriteParamWord(command,thedata);
     Hex: SerWriteParamWord(command,thedata);
     LBP: LBPWriteParamWord(command,thedata);
   end;
end;

procedure WriteParam(theaxis: byte;theparm: word;thedata : longint);
var command : word;
{ full 32 bit parameter write - 2 words }
begin
  WaitForICDFIFONotHalfFull;
  command := (theparm or WriteFlag or LongFlag or (theaxis shl AxisOffset));
   case ProtocolType of
     Bus:  BusWriteParam(command,thedata);
     Hex: SerWriteParam(command,thedata);
     LBP:  LBPWriteParam(command,thedata);
   end;
end;

procedure WriteWaitToken(theaxis: byte;theparm: word);
var command : word;
begin
  WaitForICDFIFONotHalfFull;
  command := (theparm or WaitTokenFlag or (theaxis shl AxisOffset));
   case ProtocolType of
     Bus:  BusWriteCommand(command);
     Hex: SerWriteCommand(command);
     LBP:  LBPWriteCommand(command);
   end;
end;

{$IFDEF COPROC}
procedure WriteParamDouble(theaxis: byte;theparm: word;thedata : comp);
var command : word;
{64 bit parameter write - 4 words }
begin
  WaitForICDFIFONotHalfFull;
  command := (theparm or WriteFlag or DoubleFlag or (theaxis shl AxisOffset));
  case ProtocolType of
    Bus:  BusWriteParamDouble(command,thedata);
    Hex: SerWriteParamDouble(command,thedata);
    LBP:  LBPWriteParamDouble(command,thedata);
  end;
end;
{$ENDIF}

function ReadParamWord(theaxis: byte;theparm: word):word;
var command : word;
{16 bit parameter read }
begin
  WaitForICDFIFONotHalfFull;
  command := (theparm or (theaxis shl AxisOffset));
  case ProtocolType of
    Bus: ReadParamWord := BusReadParamWord(command);
    Hex: ReadParamWord := SerReadParamWord(command);
    LBP: ReadParamWord := LBPReadParamWord(command);
  end;
end;

function ReadParam(theaxis: byte;theparm: word):longint;
var command : word;
{ 32 bit parameter read }
begin
  WaitForICDFIFONotHalfFull;
  command := (theparm or LongFlag or (theaxis shl AxisOffset));
  case ProtocolType of
    Bus:  ReadParam := BusReadParam(command);
    Hex: ReadParam := SerReadParam(command);
    LBP:  ReadParam := LBPReadParam(command);
  end;
end;


{$IFDEF COPROC}
function ReadParamDouble(theaxis: byte;theparm: word): comp;
var command : word;
{ 64 bit parameter read }
begin
  WaitForICDFIFONotHalfFull;
  command := (theparm or DoubleFlag or (theaxis shl AxisOffset));
  case ProtocolType of
    Bus:  ReadParamDouble := BusReadParamDouble(command);
    Hex: ReadParamDouble := SerReadParamDouble(command);
    LBP:  ReadParamDouble := LBPReadParamDouble(command);
  end;
end;
{$ENDIF}

procedure WriteParamWordQ(theaxis: byte;theparm: word;thedata : word);
var command : word;
{ 16 bit parameter write }
begin
  WaitForQCDFIFONotHalfFull;
  command := (theparm or WriteFlag or (theaxis shl AxisOffset));
   case ProtocolType of
     Bus:  BusWriteParamWordQ(command,thedata);
     Hex: SerWriteParamWordQ(command,thedata);
     LBP:  LBPWriteParamWordQ(command,thedata);
   end;
end;

procedure WriteParamQ(theaxis: byte;theparm: word;thedata : longint);
var command : word;
{ full 32 bit parameter write - 2 words }
begin
  WaitForQCDFIFONotHalfFull;
  command := (theparm or WriteFlag or LongFlag or (theaxis shl AxisOffset));
   case ProtocolType of
     Bus:  BusWriteParamQ(command,thedata);
     Hex: SerWriteParamQ(command,thedata);
     LBP:  LBPWriteParamQ(command,thedata);
   end;
end;

procedure WriteWaitTokenQ(theaxis: byte;theparm: word);
var command : word;
begin
  WaitForQCDFIFONotHalfFull;
  command := (theparm or WaitTokenFlag or (theaxis shl AxisOffset));
   case ProtocolType of
     Bus:  BusWriteCommandQ(command);
     Hex: SerWriteCommandQ(command);
     LBP:  LBPWriteCommandQ(command);
   end;
end;

{$IFDEF COPROC}
procedure WriteParamDoubleQ(theaxis: byte;theparm: word;thedata : comp);
var command : word;
{ 64 bit parameter write - 4 words }
begin
  WaitForQCDFIFONotHalfFull;
  command := (theparm or WriteFlag or DoubleFlag or (theaxis shl AxisOffset));
   case ProtocolType of
     Bus:  BusWriteParamDoubleQ(command,thedata);
     Hex: SerWriteParamDoubleQ(command,thedata);
     LBP:  LBPWriteParamDoubleQ(command,thedata);
   end;
end;
{$ENDIF}

function ReadParamWordQ(theaxis: byte;theparm: word):word;
var command : word;
{16 bit parameter read }
begin
  WaitForQCDFIFONotHalfFull;
  command := (theparm or (theaxis shl AxisOffset));
  case ProtocolType of
    Bus:  ReadParamWordQ := BusReadParamWordQ(command);
    Hex: ReadParamWordQ := SerReadParamWordQ(command);
    LBP:  ReadParamWordQ := LBPReadParamWordQ(command);
  end;
end;

function ReadParamQ(theaxis: byte;theparm: word):longint;
var command : word;
{ 32 bit parameter read }
begin
  WaitForQCDFIFONotHalfFull;
  command := (theparm or LongFlag or (theaxis shl AxisOffset));
  case ProtocolType of
    Bus:  ReadParamQ := BusReadParamQ(command);
    Hex: ReadParamQ := SerReadParamQ(command);
    LBP:  ReadParamQ := LBPReadParamQ(command);
  end;
end;

{$IFDEF COPROC}
function ReadParamDoubleQ(theaxis: byte;theparm: word): comp;
var command : word;
{ 64 bit parameter read }
begin
  WaitForQCDFIFONotHalfFull;
  command := (theparm or LongFlag or (theaxis shl AxisOffset));
  case ProtocolType of
    Bus:  ReadParamDoubleQ := BusReadParamDoubleQ(command);
    Hex: ReadParamDoubleQ := SerReadParamDoubleQ(command);
    LBP:  ReadParamDoubleQ := LBPReadParamDoubleQ(command);
  end;
end;
{$ENDIF}

procedure ClearICDFIFO;
var dummy : word;
begin
  if CardType = ThreeC20 then SerClearICDFIFO
  else
  begin
    WriteParamWordQ(0,IFIFOReadCountLoc,0);
    dummy := ReadParamWordQ(0,ZeroLoc);
  end;
end;

procedure ClearQCDFIFO;
var dummy : word;
begin
  if CardType = ThreeC20 then SerClearQCDFIFO
  else
  begin
    WriteParamWord(0,QFIFOReadCountLoc,0);
    dummy := ReadParamWord(0,ZeroLoc);
    { this readback is to guarantee that the FIFO is cleared }
    { before subsequent data is written (and lost ) }
  end;
end;

procedure ClearIRBFIFO;
var dummy : word;
begin
  if CardType = ThreeC20 then SerClearIRBFIFO
  else
  begin
    WriteParamWordQ(0,IFIFOWriteCountLoc,0);
    dummy := ReadParamWordQ(0,ZeroLoc);
  end;
end;

procedure ClearQRBFIFO;
var dummy : word;
begin
  if CardType = ThreeC20 then SerClearQRBFIFO
  else
  begin
    WriteParamWord(0,QFIFOWriteCountLoc,0);
    dummy := ReadParamWord(0,ZeroLoc);
  end;
end;

procedure ClearFIFOs;
begin
  ClearQCDFIFO;
  ClearIRBFIFO;
  ClearQRBFIFO;
  ClearICDFIFO;
end;

function ReadQRBFIFOSize: word;
begin
  ReadQRBFIFOSize := ReadParamWord(0,qrbfifosizeLoc);
end;

{$ENDIF FIFO}

{$IFNDEF WINDOWS}
function InitBusInterface(var message : string) : boolean;
var hstring : string;
begin
{$IFDEF FIFO}
  InitBusInterface := false;
  if SetupPCIWithMessage(PCIPort,message) then
  begin
    CardType := PCI;
    hstring := HexString(PCIPort,4);
    BusInitSoftDMC(PCIPort);
    InitBusInterface := true;
  end
  else
  begin
    hstring := HexString(PC104Port,4);
    CardType := PC104;
    message := 'Using PC104 Port '+ hstring;
    BusInitSoftDMC(PC104Port);
    InitBusInterface := true;
  end;
{$ENDIF FIFO}
end;
{$ENDIF}

function InitializeInterface(var message : string) : boolean;
begin
  InitializeInterface := false;
  ExitOnTimeout := true;
  message := 'Unable to initialize';
  SerError := false;
  case ProtocolType of
{$IFNDEF WINDOWS}
    Bus:
    begin
      if InitBusInterface(message) then InitializeInterface := true;
    end;
{$ENDIF}
    LBP:
    begin  {serial}
      if SerOpen(message) then
      begin
        if LBPSync(message) then
        begin
          InitializeInterface := true;
{$IFDEF FIFO}
          LBPInitSoftDMC(DefaultLBPBasePort);  { not clean move up }
{$ENDIF FIFO}
        end;
      end;
    end; {lbp}
    HEX:
    begin  {serial}
      if SerOpen(message) then
{$IFDEF THREEC20}
      SerMesaStart;
      SerListen(Axis);
      CardType := ThreeC20;
{$ENDIF}
      begin
        if SerSync then
        begin
          InitializeInterface := true;
          message := 'Using Hex Serial Interface';
        end else message := 'Hex Serial Communication failed !';
      end;
    end; {hex}
  end; {endcase}
end;
