/*---------------------------------------------------------------------------*/
/*
   "PCPUSRVC.H"

   Include file containing constants and structures for use with Mesa
	 Electronics PMPC-2 CPU card public service functions.

   Version 1.0, Friday October 23, 1998 -- 23:31:33.

   Copyright (c) 1998 by Mesa Electronics.
*/
/*---------------------------------------------------------------------------*/
/*
   Revision history.

   1) Version 1.0, Friday October 23, 1998 -- 23:31:33.

      Code frozen for version 1.0.
*/
/*---------------------------------------------------------------------------*/

#if !defined H_PCPUSRVC
#define H_PCPUSRVC

/*---------------------------------------------------------------------------*/

/* Enumerations.
*/

/* Function numbers that are used to access the ROMBIOS's public service
	 functions.
*/
typedef enum publicservicefuncs
{	/* The following are code revision level 0 functions.
	*/
	F_PUBSRVCINFOQ = 0,          /* Get code revision level, etc. */
	F_PUBSRVCXABLEKB,            /* Xable PC keyboard input. */
	F_PUBSRVCXABLESKB,           /* Xable serial keyboard input. */
	F_PUBRESERVED1,              /* Reserved function code. */
	F_PUBSRVCUSERINFOLENQ,       /* Get number of bytes of user information. */
	F_PUBSRVCUSERINFOQ,          /* Get user information. */
	F_PUBSRVCNUMCONFIGWORDSQ,    /* Inquire number of user configuration words available. */
	F_PUBSRVCCONFIGWORDREAD,     /* Read configuration word. */
	F_PUBSRVCCONFIGWORDWRITE,    /* Write configuration word. */
	F_PUBSRVCCPUREVQ,            /* Get CPU card revision number. */
	F_PUBSRVCATODTYPE,           /* Get local A-to-D converter type. */
	F_PUBSRVCATODRAWREAD,        /* Raw read of A-to-D converter channel. */
	F_PUBSRVCDISPCONTRAST,       /* Set display contrast. */
	F_PUBSRVCDISPCONTRASTQ,      /* Get current display contrast. */
	F_PUBSRVCBACKLITEBRITENESS,  /* Set display backlight brightness. */
	F_PUBSRVCBACKLITEBRITENESSQ, /* Get display backlight brightness. */
	F_PUBSRVCDISPTIMEOUT,        /* Set backlightshut-off timeout. */
	F_PUBSRVCDISPTIMEOUTQ,       /* Get backlightshut shut-off timeout. */
	F_PUBSRVCVIDEODEST,          /* Select video output destination. */
	F_PUBSRVCVIDEODESTQ,         /* Get video output destination. */
	F_PUBSRVCCONSOLEBAUDRATE,    /* Set serial console COM port baud rate. */
	F_PUBSRVCCONSOLEBAUDRATEQ,   /* Get serial console COM port baud rate. */
	F_PUBSRVCVIDEOVECTORQ,       /* Return entry point for specified video handler. */

	NUMPUBSRVCFUNCS
} publicservicefuncs ;


/* Basic error codes for public service functions. */

typedef enum publicserviceerrs
{
	E_PUBSRVCERRNONE = 0, /* No error. */
	E_PUBSRVCERRBADFUNC,  /* Invalid function number. */
	E_PUBSRVCERRBASE,     /* Base error code number. */
} publicserviceerrs ;

/* One of the private system services is needed */

typedef enum servicefuncs
{
F_SYSSETUPSAVE = 2        /* Write system configuration image to non-volatile storage. */
} servicefuncs;

/* Constants used for specifying membrane keyboard events. */

typedef enum membraneeventid
{
	KSTATE_NONE = 0, /* No event. */
	KSTATE_MAKE,     /* Key make event. */
	KSTATE_BREAK,    /* Key break event. */
	KSTATE_REPEAT,   /* Key repeat event. */

	NUMKSTATES
} membraneeventid ;

/*---------------------------------------------------------------------------*/

/* Structures.
*/
/* Structure used to pass a command to, and receive completion status from, a
     ROMBIOS's public functions.  This structure is always the first element
     of any public function command structure.

   When the special functions entry point is called, CX:BX must be loaded
     with a far pointer to one of the following structures that begins with a
     pfuncshdr structure.  (CX contains the segment, BX contains the offset.)
*/
typedef struct pfuncshdr
{
	unsigned short commandCode ; /* The command code from the list of syscntrlfuncs. */
	unsigned short errorCode ;   /* The result code from the list of syscntrlerrs. */
} pfuncshdr ;


/* Structure used to inquire whether the public functions code is available.
     (pubfuncsavailq.commandHeader.commandCode == F_PUBSRVCINFOQ.)
*/
typedef struct pubfuncsavailq
{
	pfuncshdr commandHeader ; /* Command/status information. */

	unsigned char toggleByte ;     /* Gets inverted if there's code to answer. */
	unsigned short revisionLevel ; /* Code revision level. */
} pubfuncsavailq ;


/* Structure used to xable keyboard
	 input. (pubxablekb.commandHeader.commandCode == F_PUBSRVCXABLEKB,
	 F_PUBSRVCXABLESKB.)
*/
typedef struct pubxablekb
{
	pfuncshdr commandHeader ; /* Command/status information. */

	unsigned char getSettings ; /* Non-zero to get state, zero to set. */

	unsigned char enableKB ;    /* Xable flag. (Setting state returns previous state.) */
} pubxablekb ;


/* Structure used to read user information length.
     (pubuserinfolenq.commandHeader.commandCode == F_PUBSRVCUSERINFOLENQ.)
*/
typedef struct pubuserinfolenq
{
	pfuncshdr commandHeader ; /* Command/status information. */

	unsigned short userInfoLen ; /* Bytes of storage required for user information. */
} pubuserinfolenq ;


/* Structure used to read user information.
     (pubuserinfoq.commandHeader.commandCode == F_PUBSRVCUSERINFOQ.)
*/
typedef struct pubuserinfoq
{
	pfuncshdr commandHeader ; /* Command/status information. */

	unsigned short bytesToRead ;     /* Number of bytes of user information to read. -1 for all bytes. */
	unsigned char far *userInfoPtr ; /* Pointer to user information destination. */
} pubuserinfoq ;


/* Structure used to inquire the number of system configuration words
	 available for customer use.
     (pubconfigwordsq.commandHeader.commandCode == F_PUBSRVCNUMCONFIGWORDSQ.)
*/
typedef struct pubconfigwordsq
{
	pfuncshdr commandHeader ; /* Command/status information. */

	unsigned short numWords ; /* Number of available configuration words. */
} pubconfigwordsq ;


/* Structure used to read or write system configuration words.
     (pubconfigword.commandHeader.commandCode == F_PUBSRVCCONFIGWORDREAD or
     F_PUBSRVCCONFIGWORDWRITE.)

   User configuration words are uncommitted non-volatile storage that may be
     used for customer-specific applications.  Valid configuration word
     indices are 0 to (pubconfigwordsq.numWords - 1), inclusive.  The
     specified configuration word is written to non-volatile storage upon
     invocation of F_PUBSRVCCONFIGWORDWRITE, rather than when the system
     configuration is saved, so the customer's software must manage wear on
     those systems that use EEPROM for non-volatile storage.
*/
typedef enum pubconfigworderrs
{
	E_PUBCONFIGWORDINDEX = E_PUBSRVCERRBASE, /* Invalid configuration word index. */
	E_PUBCONFIGWORDSHARE,                    /* Shared resource unavailable. Try again. */
	E_PUBCONFIGWORDWRITE,                    /* Write verification failure. */

	NUMPUBCONFIGWORDERRS
} pubconfigworderrs ;

typedef struct pubconfigword
{
	pfuncshdr commandHeader ; /* Command/status information. */

	unsigned short wordIndex ; /* The index of the word to be written. */
	unsigned short theWord ;   /* The word. */
} pubconfigword ;


/* Structure used to inquire about the CPU card's hardware revision level.
     (pubcpurevinfo.commandHeader.commandCode == F_PUBSRVCCPUREVQ.)
*/
typedef struct pubcpurevinfo
{
	pfuncshdr commandHeader ; /* Command/status information. */

	unsigned char cpuRevNum ; /* 0 for rev. A, 1 for rev. B, etc. */
} pubcpurevinfo ;


/* Structure used to read the local digital-to-analog converter type.
     (puba2dtypeinfo.commandHeader.commandCode == F_PUBSRVCATODTYPE.)
*/
typedef struct puba2dtypeinfo
{
	pfuncshdr commandHeader ; /* Command/status information. */

	char far *a2dTypeStringPtr ; /* Pointer to upper case '\0'-terminated string containing converter part number. */
	unsigned char numChannels ;  /* Number of accessible channels. */
} puba2dtypeinfo ;


/* Structure used to read an A-to-D converter channel.
     (puba2dread.commandHeader.commandCode == F_PUBSRVCATODRAWREAD.

   Note that setting .isTimeCritical to non-zero will cause the function to
     return with E_PUBA2DREADTIMEOUT immediately if the A-to-D converter is
	 busy.  If .isTimeCritical is set to zero, the function will wait until
	 it has access to the converter before returning.

  All return values are scaled to the range 0...65535 -- for a 10-bit
    converter, bits 0...5 contain zero; for a 12 bit converter, bits 0...3
	contain zero, and for a 16-bit converter all bits are significant.
*/

/* Dedicated A-to-D converter channels.
*/
#define A2DCHANTEMP   9  /* Temperature channel. */
#define A2DCHANVCC    8  /* VCC voltage channel. */
#define A2DCHANGROUND 10 /* Ground channel. */
#define A2DCHANTEST   11 /* Converter test channel. */

typedef enum puba2dreaderrs
{
	E_PUBA2DREADTIMEOUT = E_PUBSRVCERRBASE, /* The converter is busy. */
	E_PUBA2DREADCHANNEL, /* Invalid converter channel number. */

	NUMPUBA2DREADERRS
} puba2dreaderrs ;

typedef struct puba2dread
{
	pfuncshdr commandHeader ; /* Command/status information. */

	unsigned char isTimeCritical ; /* Non-zero will give up immediately if the A-to-D converter is busy. */

	unsigned char channelNumber ;  /* Number of the converter channel to be read. */
	unsigned short a2dData ;       /* Returned conversion value. */
} puba2dread ;


/* Structure used to read/write display contrast.
     (pubdispcontrast.commandHeader.commandCode == F_PUBSRVCDISPCONTRAST or
	 F_PUBSRVCDISPCONTRASTQ.)
*/
typedef struct pubdispcontrast
{
	pfuncshdr commandHeader ; /* Command/status information. */

	/* The display contrast value has a full scale value of 65535 and a minimum
	     value of 0.  A contrast setting of 32767 therefore gives a setting of
	     about 50 percent full scale.
	*/
	unsigned short displayContrastPercent ;
} pubdispcontrast ;


/* Structure used to read/write display backlight brightness.
	 (pubbacklightbriteness.commandHeader.commandCode ==
	 F_PUBSRVCBACKLITEBRITENESS or F_PUBSRVCBACKLITEBRITENESSQ.)
*/
typedef struct pubbacklightbriteness
{
	pfuncshdr commandHeader ; /* Command/status information. */

	/* The display backlight brightness value has a full scale value of 65535 and
	     a minimum value of 0.  A contrast setting of 32767 therefore gives a
		 setting of about 50 percent full scale.
	*/
	unsigned short backlightLevelPercent ;
} pubbacklightbriteness ;


/* Structure used to read/write display shut-off timeout.
	 (pubdisptimeout.commandHeader.commandCode == F_PUBSRVCDISPTIMEOUT or
	 F_PUBSRVCDISPTIMEOUTQ.)
*/
typedef enum pubdisptimeouterrs
{
	E_PUBDISPTIMEOUT = E_PUBSRVCERRBASE, /* Invalid timeout. */

	NUMPUBDISPTIMEOUTERRS
} pubdisptimeouterrs ;

typedef struct pubdisptimeout
{
	pfuncshdr commandHeader ; /* Command/status information. */

	unsigned char timeOut ; /* Backlight shutoff timeout in 25.6 second ticks. (1...32.) */
} pubdisptimeout ;


/* Structure used to inquire about/adjust display output routing.
    (pubvideodest.commandHeader.commandCode == F_PUBSRVCVIDEODEST or
	F_PUBSRVCVIDEODESTQ.)
*/

/* Constants used for specifying various video output destinations.
*/
typedef enum pubvideodests
{
	PUBVIDEODEST_VIDEO = 0, /* Send video output to standard video. */
	PUBVIDEODEST_SERIAL,    /* Send video output to local serial port. */
	PUBVIDEODEST_LCD,       /* Send video output to local LCD video. */
	PUBVIDEODEST_STUB,      /* Send video output to black hole. */

	NUMPUBVIDEODESTS
} pubvideodests ;

/* Additional error codes returned by F_PUBSRVCVIDEODEST and
	 F_PUBSRVCVIDEODESTQ.
*/
typedef enum pubvideodesterrs
{
	E_PUBVIDDEST = E_PUBSRVCERRBASE, /* Invalid video destination. */

	NUMPUBVIDDESTERRS
} pubvideodesterrs ;

typedef struct pubvideodest
{
	pfuncshdr commandHeader ; /* Command/status information. */

	pubvideodests videoDest ; /* Where video output is to go to. (From enum pubvideodests.) */
} pubvideodest ;


/* Structure used to inquire about/adjust serial console COM port baud rate.
     (pubconsolebaudrate.commandHeader.commandCode ==
	 F_PUBSRVCCONSOLEBAUDRATE or F_PUBSRVCCONSOLEBAUDRATEQ.)
*/

/* Baud rate selectors.
*/
typedef enum pubconsolebaudrates
{
	PUBCONSOLEBAUDRATE_110 = 0, /* 110 Baud. */
	PUBCONSOLEBAUDRATE_150,
	PUBCONSOLEBAUDRATE_300,
	PUBCONSOLEBAUDRATE_600,
	PUBCONSOLEBAUDRATE_1200,
	PUBCONSOLEBAUDRATE_1800,
	PUBCONSOLEBAUDRATE_2000,
	PUBCONSOLEBAUDRATE_2400,
	PUBCONSOLEBAUDRATE_3600,
	PUBCONSOLEBAUDRATE_4800,
	PUBCONSOLEBAUDRATE_7200,
	PUBCONSOLEBAUDRATE_9600,
	PUBCONSOLEBAUDRATE_19200,
	PUBCONSOLEBAUDRATE_38400,
	PUBCONSOLEBAUDRATE_57600,
	PUBCONSOLEBAUDRATE_115200,  /* 115.2 Kbaud. */

	NUMPUBCONSOLEBAUDRATES
} pubconsolebaudrates ;

/* Additional error codes returned by F_PUBSRVCCONSOLEBAUDRATE and
     F_PUBSRVCCONSOLEBAUDRATEQ.
*/
typedef enum pubconsolebaudrateerrs
{
	E_PUBCONSOLEBAUDRATE = E_PUBSRVCERRBASE, /* Invalid baud rate selector. */

	NUMPUBCONSOLEBAUDRATEERRS
} pubconsolebaudrateerrs ;

typedef struct pubconsolebaudrate
{
	pfuncshdr commandHeader ; /* Command/status information. */

	pubconsolebaudrates baudSelector ; /* Baud rate selector. One of enum pubconsolebaudrate. Set returns previous selector. */
} pubconsolebaudrate ;


/* Structure used to get the specified video code vector.
     (pubvideovectorq.commandHeader.commandCode == F_PUBSRVCVIDEOVECTORQ.)
*/

/* Video vector selectors.
*/
typedef enum pubvideovectorselect
{
	PUBVIDEOVECTORSELECT_VIDEO = 0,
	PUBVIDEOVECTORSELECT_SERIAL,
	PUBVIDEOVECTORSELECT_LCD,
	PUBVIDEOVECTORSELECT_STUB,

	NUMPUBVIDEOVECTORSELECTS
} pubvideovectorselect ;

/* Additional error codes returned by F_PUBSRVCVIDEOVECTORQ.
*/
typedef enum pubvideovectorerrs
{
	E_PUBVIDEOVECTOR = E_PUBSRVCERRBASE, /* No such video vector. */

	NUMPUBVIDEOVECTORERRS
} pubvideovectorerrs ;

typedef struct pubvideovectorq
{
	pfuncshdr commandHeader ; /* Command/status information. */

	pubvideovectorselect whichHandler ; /* Handler specifier. */
	void far *codeEntryPoint ;          /* Address of the specified video output handler. 0 if the handler is undefined. */
} pubvideovectorq ;

/*---------------------------------------------------------------------------*/

#endif
