/*---------------------------------------------------------------------------*/
/*
   "TTYDEMO.C"

   Demonstration program for the Mesa Electronics 4C22 when used with a
     liquid crystal display.

   Version 1.1, Tuesday February 22, 1994 - 15:48:00.
*/
/*---------------------------------------------------------------------------*/
/*
   Compiler: Borland C++, version 3.1.
*/
/*---------------------------------------------------------------------------*/
/*
   Revision history.

   1) Version 1.0, Wednesday December 8, 1993 - 19:25:25.

      Code frozen for version 1.0.


   2) Version 1.1, Tuesday February 22, 1994 - 15:48:00.
	  Changes from version 1.0:
	  1) setttyregion() now erases the non-TTY region of the screen manually.
		 4C22 ROMBIOS versions prior to 1.0b did this implicitely when the
		 TTY region characteristics were modified.  This behavior was
		 discontinued with ROMBIOS version 1.0b.
*/
/*---------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <dos.h>
#pragma option -a- /* (Structures must be byte-aligned.) */
#include "sint1a.h"
#include "sint10.h"

/*---------------------------------------------------------------------------*/

#define TTYUPPERLEFTX 5   /* X coordinate of upper left-hand corner of TTY region. */
#define TTYUPPERLEFTY 6   /* Y coordinate of upper left-hand corner of TTY region. */
#define TTYDIMX       115 /* X dimension of TTY region. */
#define TTYDIMY       50  /* Y dimension of TTY region. */

/*---------------------------------------------------------------------------*/

/* Local externals.
*/
static unsigned char BackGroundPattern[8] = { 0xEF, 0xDF, 0xBF, 0x7F, 0xBE, 0xDD, 0xEB, 0xF7 } ; /* (Patterns have 8 bytes.) */

/*---------------------------------------------------------------------------*/


/* Function: sint10ctrlcall
   Purpose: Call special interrupt 0x10 control functions.
   Used by: Anyone.
   Returns: Pass/fail status.
   Notes:
   Revision history:
     1) Wednesday December 8, 1993 - 19:25:25.
*/

signed sint10ctrlcall(void far *parmptr)

{	asm {
				mov		ah,F_SPCLVIDCNTRL
				mov		bx,word ptr parmptr[0]	// Offset of far pointer.
				mov		cx,word ptr parmptr[2]	// Segment of far ptr.
				int		010H					// Invoke the function.
	}
	return (((((int10cmndhdr far *)parmptr) -> errorCode) != E_DISPNONE) ? -1 : 0) ;
}

/*---------------------------------------------------------------------------*/

/* Function: sint10grfxcall
   Purpose: Call special interrupt 0x10 control functions.
   Used by: Anyone.
   Returns: Pass/fail status.
   Notes:
   Revision history:
     1) Wednesday December 8, 1993 - 19:25:25.
*/

signed sint10grfxcall(void far *parmptr)

{	asm {
				mov		ah,F_SPCLVIDGRFX
				mov		bx,word ptr parmptr[0]	// Offset of far pointer.
				mov		cx,word ptr parmptr[2]	// Segment of far ptr.
				int		010H					// Invoke the function.
	}
	return (((((int10cmndhdr far *)parmptr) -> errorCode) != E_GRFXNONE) ? -1 : 0) ;
}

/*---------------------------------------------------------------------------*/

/* Function: sint1actrlcall
   Purpose: Call special interrupt 0x1A control functions.
   Used by: Anyone.
   Returns: Pass/fail status.
   Notes:
   Revision history:
     1) Wednesday December 8, 1993 - 19:25:25.
*/

signed sint1actrlcall(void far *parmptr)

{	asm {
				mov		ah,F_SPCLSYSCNTRL
				mov		bx,word ptr parmptr[0]	// Offset of far pointer.
				mov		cx,word ptr parmptr[2]	// Segment of far ptr.
				int		01AH					// Invoke the function.
	}
	return (((((int1acmndhdr far *)parmptr) -> errorCode) != E_SYSERRNONE) ? -1 : 0) ;
}

/*---------------------------------------------------------------------------*/

/* Function: havespecialint10codeq
   Purpose: Determine whether or not interrupt 0x10 special function code is
     avaible to the programmer.
   Used by: Anyone.
   Returns: Available/not available status.
   Notes:
     -In general, this function need not be called if the CPU is known to
	    support the special interrupt 0x10 functions.
   Revision history:
     1) Wednesday December 8, 1993 - 19:25:25.
*/

int havespecialint10codeq(void)

{	ctrlavailqinfo availinfo ;


	(availinfo . commandHeader . commandCode) = F_DISPCNTRLINFOQ ;
	(availinfo . toggleByte) = 0x55 ;
	if(sint10ctrlcall(&availinfo) < 0)
	{	return !!0 ; /* Error - code not available. */
	}
	else if((availinfo . toggleByte) != 0xAA)
	{	return !!0 ; /* No toggle of toggle byte - code not available. */
	}
	else
	{	return !0 ; /* Code available. */
	}
}

/*---------------------------------------------------------------------------*/

/* Function: havespecialint1acodeq
   Purpose: Determine whether or not interrupt 0x1A special function code is
     avaible to the programmer.
   Used by: Anyone.
   Returns: Available/not available status.
   Notes:
     -In general, this function need not be called if the CPU is known to
	    support the special interrupt 0x1A functions.
   Revision history:
     1) Wednesday December 8, 1993 - 19:25:25.
*/

int havespecialint1acodeq(void)

{	sysfuncsavailq availinfo ;


	(availinfo . commandHeader . commandCode) = F_SYSCNTRLINFOQ ;
	(availinfo . toggleByte) = 0x55 ;
	if(sint1actrlcall(&availinfo) < 0)
	{	return !!0 ; /* Error - code not available. */
	}
	else if((availinfo . toggleByte) != 0xAA)
	{	return !!0 ; /* No toggle of toggle byte - code not available. */
	}
	else
	{	return !0 ; /* Code available. */
	}
}

/*---------------------------------------------------------------------------*/

/* Function: setttyregion
   Purpose: Change the dimensions of the TTY region.
   Used by: Anyone.
   Returns: Nothing.
   Notes:
   Revision history:
     1) Wednesday December 8, 1993 - 19:25:25.
*/

static void setttyregion(void)

{	sysvidreroute rerouteinfo ;

	dispavailqinfo availinfo ;

	ttyregninfo ttyinfo ;

	eraserectinfo eraseinfo ;

	fillpatinfo patinfo ;


	/* First, see if we have access to the special interrupt 0x1A functions that
	     we need to reroute video.  (In an environment known to support the
	     special interrupt 0x1A functions, this check need not be performed.)
	*/
	if(!havespecialint1acodeq())
	{	fprintf(stderr, "\n\aSpecial interrupt 0x1A function code is not available.\n") ;

		exit(1) ;
	}

	/* Video rerouting is available; try connecting video output to LCD output. */
	(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ;
	(rerouteinfo . videoDest) = VIDDEST_LCD ; /* Video --> LCD. */
	if(sint1actrlcall(&rerouteinfo) < 0)
	{	fprintf(stderr, "\n\aCouldn't reroute video output to LCD display.\n") ;

		exit(1) ;
	}

	/* Video output functions were successfully rerouted to LCD output functions.
	     We must now determine whether we have access to the functions we want.
	     (In an environment known to support the special interrupt 0x10
	     functions, this check need not be performed.)
	*/
	if(!havespecialint10codeq())
	{	/* Oops!  Our video output functions don't seem to be available.  We must now
		     switch back to standard video output in order to be able to report the
		     error condition to the user.
		*/
		(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ; /* (Redundant.) */
		(rerouteinfo . videoDest) = VIDDEST_VIDEO ; /* Video --> video. */
		sint1actrlcall(&rerouteinfo) ; /* We hope this works, as there's no other way out of this mess. */

		fprintf(stderr, "\n\aSpecial interrupt 0x10 function code is not available.\n") ;

		exit(1) ;
	}

	/* See if the LCD display is active and available. */
	(availinfo . commandHeader . commandCode) = F_GRFXDISPINFOQ ;
	(availinfo . toggleByte) = 0x55 ;
	if(sint10grfxcall(&availinfo) < 0)
	{	/* This should never happen. */
		(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ;
		(rerouteinfo . videoDest) = VIDDEST_VIDEO ; /* Video --> video. */
		sint1actrlcall(&rerouteinfo) ;

		fprintf(stderr, "\n\aCan't determine display availability.\n") ;

		exit(1) ;
	}
	if((availinfo . toggleByte) != 0xAA)
	{	(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ;
		(rerouteinfo . videoDest) = VIDDEST_VIDEO ; /* Video --> video. */
		sint1actrlcall(&rerouteinfo) ;

		fprintf(stderr, "\n\aGraphics code doesn't respond to queries.\n") ;

		exit(1) ;
	}
	if(!(availinfo . isAvailable))
	{	(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ;
		(rerouteinfo . videoDest) = VIDDEST_VIDEO ; /* Video --> video. */
		sint1actrlcall(&rerouteinfo) ;

		fprintf(stderr, "\n\aDisplay isn't available.\n") ;

		exit(1) ;
	}

	/* The LCD display is active and available.

	   We first change the display's non-TTY region pattern so the new TTY region
	     can actually be seen.  (Note that patterns are stored in the ROMBIOS's
	     private memory, so the source pattern need not be preserved after a call
	     to F_GRFXPATSET.)
	*/
	(patinfo . commandHeader . commandCode) = F_GRFXPATSET ;
	(patinfo . patPtr) = BackGroundPattern ;
	if(sint10grfxcall(&patinfo) < 0)
	{	(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ;
		(rerouteinfo . videoDest) = VIDDEST_VIDEO ; /* Video --> video. */
		sint1actrlcall(&rerouteinfo) ;

		fprintf(stderr, "\n\aUnable to set display pattern.\n") ;

		exit(1) ;
	}

	/* Now change the dimensions of the TTY region.  (And therefore the non-TTY
	     region.)

	   We start by getting the current TTY region information, as we need to know
	     what the current font-related information is in order to avoid having to
	     make it up ourselves.

	   When the new TTY region is created, the new new non-TTY region background
	     pattern will appear in the vacated areas, if any.
	*/
	(ttyinfo . commandHeader . commandCode) = F_GRFXTTYINFOGET ;
	if(sint10grfxcall(&ttyinfo) < 0)
	{	(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ;
		(rerouteinfo . videoDest) = VIDDEST_VIDEO ; /* Video --> video. */
		sint1actrlcall(&rerouteinfo) ;

		fprintf(stderr, "\n\aCan't get current TTY information.\n") ;

		exit(1) ;
	}

	/* Fill the current TTY region with the non-TTY region fill patter before we
	     reduce the TTY region's size.  This has the effect of filling the areas
	     of the screen that will be vacated when the size of the TTY region is
	     reduced with the non-TTY region fill pattern.
	*/
	(eraseinfo . commandHeader . commandCode) = F_GRFXERASEREGN ;
	(eraseinfo . theRegion . minCoord) = (ttyinfo . ttyUpperLeft) ;
	(eraseinfo . theRegion . maxCoord . x) = ((eraseinfo . theRegion . minCoord . x) + ((ttyinfo . ttyDim . x) - 1)) ;
	(eraseinfo . theRegion . maxCoord . y) = ((eraseinfo . theRegion . minCoord . y) + ((ttyinfo . ttyDim . y) - 1)) ;
	sint10grfxcall(&eraseinfo) ; /* (We don't bother to check for errors.) */

	/* Now change the TTY region location and dimensions.  We don't mess with the
	     current font-related information.

	   Note that setting the TTY region information has the side-effect of
	     clearing the display.
	*/
	(ttyinfo . commandHeader . commandCode) = F_GRFXTTYINFOSET ;
	(ttyinfo . ttyUpperLeft . x) = TTYUPPERLEFTX ;
	(ttyinfo . ttyUpperLeft . y) = TTYUPPERLEFTY ;
	(ttyinfo . ttyDim . x) = TTYDIMX ;
	(ttyinfo . ttyDim . y) = TTYDIMY ;
	if(sint10grfxcall(&ttyinfo) < 0)
	{	/* The TTY region probably doesn't fit within the physical display
		     dimensions.  (We're too lazy to check it out.)
		*/
		(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ;
		(rerouteinfo . videoDest) = VIDDEST_VIDEO ; /* Video --> video. */
		sint1actrlcall(&rerouteinfo) ;

		fprintf(stderr, "\n\aCan't set new TTY information.\n") ;

		exit(1) ;
	}
}

/*---------------------------------------------------------------------------*/

/* Function: main
   Purpose: Program entry point.
   Used by: Program launcher.
   Returns: Never.
   Notes:
   Revision history:
     1) Wednesday December 8, 1993 - 19:25:25.
*/

int main(unsigned argc, char *argv[])

{	setttyregion() ;

	exit(0) ;
}

/*---------------------------------------------------------------------------*/
