/*---------------------------------------------------------------------------*/
/*
   "TRSTDEMO.C"

   Demonstration program for the Mesa Electronics FPCX when used with a
     liquid crystal display.  Adjusts display backlight intensity and
	 contrast.

   Version 1.0, Sunday December 7, 1997 -- 20:11:34.
*/
/*---------------------------------------------------------------------------*/
/*
   Compiler: Borland C++, version 3.1.
*/
/*---------------------------------------------------------------------------*/
/*
   Revision history.

   1) Version 1.0, Sunday December 7, 1997 -- 20:11:34.

      Code frozen for version 1.0.
*/
/*---------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <dos.h>
#pragma option -a- /* (Structures must be byte-aligned.) */
#include "sint1a.h"
#include "sint10.h"

/*---------------------------------------------------------------------------*/

/* Function: sint10ctrlcall
   Purpose: Call special interrupt 0x10 control functions.
   Used by: Anyone.
   Returns: Pass/fail status.
   Notes:
   Revision history:
	 1) Sunday December 7, 1997 -- 20:11:34.
*/

signed sint10ctrlcall(void far *parmptr)

{
	asm {
				mov		ah,F_SPCLVIDCNTRL
				mov		bx,word ptr parmptr[0]	// Offset of far pointer.
				mov		cx,word ptr parmptr[2]	// Segment of far ptr.
				int		010H					// Invoke the function.
	}
	return (((((int10cmndhdr far *)parmptr) -> errorCode) != E_DISPNONE) ? -1 : 0) ;
}

/*---------------------------------------------------------------------------*/

/* Function: sint10grfxcall
   Purpose: Call special interrupt 0x10 control functions.
   Used by: Anyone.
   Returns: Pass/fail status.
   Notes:
   Revision history:
	 1) Sunday December 7, 1997 -- 20:11:34.
*/

signed sint10grfxcall(void far *parmptr)

{
	asm {
				mov		ah,F_SPCLVIDGRFX
				mov		bx,word ptr parmptr[0]	// Offset of far pointer.
				mov		cx,word ptr parmptr[2]	// Segment of far ptr.
				int		010H					// Invoke the function.
	}
	return (((((int10cmndhdr far *)parmptr) -> errorCode) != E_GRFXNONE) ? -1 : 0) ;
}

/*---------------------------------------------------------------------------*/

/* Function: sint1actrlcall
   Purpose: Call special interrupt 0x1A control functions.
   Used by: Anyone.
   Returns: Pass/fail status.
   Notes:
   Revision history:
	 1) Sunday December 7, 1997 -- 20:11:34.
*/

signed sint1actrlcall(void far *parmptr)

{
	asm {
				mov		ah,F_SPCLSYSCNTRL
				mov		bx,word ptr parmptr[0]	// Offset of far pointer.
				mov		cx,word ptr parmptr[2]	// Segment of far ptr.
				int		01AH					// Invoke the function.
	}
	return (((((int1acmndhdr far *)parmptr) -> errorCode) != E_SYSERRNONE) ? -1 : 0) ;
}

/*---------------------------------------------------------------------------*/

/* Function: havespecialint10codeq
   Purpose: Determine whether or not interrupt 0x10 special function code is
     avaible to the programmer.
   Used by: Anyone.
   Returns: Available/not available status.
   Notes:
     -In general, this function need not be called if the CPU is known to
	    support the special interrupt 0x10 functions.
   Revision history:
	 1) Sunday December 7, 1997 -- 20:11:34.
*/

int havespecialint10codeq(void)

{
	ctrlavailqinfo availinfo ;


	(availinfo . commandHeader . commandCode) = F_DISPCNTRLINFOQ ;
	(availinfo . toggleByte) = 0x55 ;
	if(sint10ctrlcall(&availinfo) < 0)
	{
		return !!0 ; /* Error - code not available. */
	}
	else if((availinfo . toggleByte) != 0xAA)
	{
		return !!0 ; /* No toggle of toggle byte - code not available. */
	}
	else
	{
		return !0 ; /* Code available. */
	}
}

/*---------------------------------------------------------------------------*/

/* Function: havespecialint1acodeq
   Purpose: Determine whether or not interrupt 0x1A special function code is
     avaible to the programmer.
   Used by: Anyone.
   Returns: Available/not available status.
   Notes:
     -In general, this function need not be called if the CPU is known to
	    support the special interrupt 0x1A functions.
   Revision history:
	 1) Sunday December 7, 1997 -- 20:11:34.
*/

int havespecialint1acodeq(void)

{
	sysfuncsavailq availinfo ;


	(availinfo . commandHeader . commandCode) = F_SYSCNTRLINFOQ ;
	(availinfo . toggleByte) = 0x55 ;
	if(sint1actrlcall(&availinfo) < 0)
	{
		return !!0 ; /* Error - code not available. */
	}
	else if((availinfo . toggleByte) != 0xAA)
	{
		return !!0 ; /* No toggle of toggle byte - code not available. */
	}
	else
	{
		return !0 ; /* Code available. */
	}
}

/*---------------------------------------------------------------------------*/

/* Function: check4lcddisplay
   Purpose: Make sure the LCD display is operational.
   Used by: Anyone.
   Returns: Nothing.
   Notes:
   Revision history:
	 1) Sunday December 7, 1997 -- 20:11:34.
*/

static void waitticks(unsigned howlong)

{
	unsigned long currtime, prevtime, stoptime ;

	static const unsigned long far *ticktimerptr = MK_FP(0x0040, 0x006C) ;


	disable() ;
	prevtime = *ticktimerptr ;
	stoptime = (prevtime + (unsigned long)howlong) ;
	for( ; ; )
	{
		disable() ;
		currtime = *ticktimerptr ;
		if(currtime < prevtime)
		{
			if(stoptime < 1573040UL)
			{
				break ;
			}
			stoptime -= 1573040UL ;
		}
		enable() ;

		if(currtime >= stoptime)
		{
			break ;
		}

		prevtime = currtime ;
	}

	enable() ;
}

/*---------------------------------------------------------------------------*/

/* Function: check4lcddisplay
   Purpose: Make sure the LCD display is operational.
   Used by: Anyone.
   Returns: Nothing.
   Notes:
   Revision history:
	 1) Sunday December 7, 1997 -- 20:11:34.
*/

static void check4lcddisplay(void)

{
	sysvidreroute rerouteinfo ;

	dispavailqinfo availinfo ;


	/* First, see if we have access to the special interrupt 0x1A functions that
	     we need to reroute video.  (In an environment known to support the
	     special interrupt 0x1A functions, this check need not be performed.)
	*/
	if(!havespecialint1acodeq())
	{
		fprintf(stderr, "\n\aSpecial interrupt 0x1A function code is not available.\n") ;

		exit(1) ;
	}

	/* Video rerouting is available; try connecting video output to LCD output.
	*/
	(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ;
	(rerouteinfo . videoDest) = VIDDEST_LCD ; /* Video --> LCD. */
	if(sint1actrlcall(&rerouteinfo) < 0)
	{
		fprintf(stderr, "\n\aCouldn't reroute video output to LCD display.\n") ;

		exit(1) ;
	}

	/* Video output functions were successfully rerouted to LCD output functions.
	     We must now determine whether we have access to the functions we want.
	     (In an environment known to support the special interrupt 0x10
	     functions, this check need not be performed.)
	*/
	if(!havespecialint10codeq())
	{	/* Oops!  Our video output functions don't seem to be available.  We must now
		     switch back to standard video output in order to be able to report the
		     error condition to the user.
		*/
		(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ; /* (Redundant.) */
		(rerouteinfo . videoDest) = VIDDEST_VIDEO ; /* Video --> video. */
		sint1actrlcall(&rerouteinfo) ; /* We hope this works, as there's no other way out of this mess. */

		fprintf(stderr, "\n\aSpecial interrupt 0x10 function code is not available.\n") ;

		exit(1) ;
	}

	/* See if the LCD display is active and available.
	*/
	(availinfo . commandHeader . commandCode) = F_GRFXDISPINFOQ ;
	(availinfo . toggleByte) = 0x55 ;
	if(sint10grfxcall(&availinfo) < 0)
	{	/* This should never happen.
		*/
		(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ;
		(rerouteinfo . videoDest) = VIDDEST_VIDEO ; /* Video --> video. */
		sint1actrlcall(&rerouteinfo) ;

		fprintf(stderr, "\n\aCan't determine display availability.\n") ;

		exit(1) ;
	}
	if((availinfo . toggleByte) != 0xAA)
	{
		(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ;
		(rerouteinfo . videoDest) = VIDDEST_VIDEO ; /* Video --> video. */
		sint1actrlcall(&rerouteinfo) ;

		fprintf(stderr, "\n\aGraphics code doesn't respond to queries.\n") ;

		exit(1) ;
	}
	if(!(availinfo . isAvailable))
	{
		(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ;
		(rerouteinfo . videoDest) = VIDDEST_VIDEO ; /* Video --> video. */
		sint1actrlcall(&rerouteinfo) ;

		fprintf(stderr, "\n\aDisplay isn't available.\n") ;

		exit(1) ;
	}
}

/*---------------------------------------------------------------------------*/

/* Function: gettrast
   Purpose: Read the current display contrast setting.
   Used by: Anyone.
   Returns: Nothing.
   Notes:
   Revision history:
	 1) Sunday December 7, 1997 -- 20:11:34.
*/

static unsigned short gettrast(void)

{
	contrastinfo trastinfo ;


	(trastinfo . commandHeader . commandCode) = F_DISPCONTRASTGET ;
	if(sint10ctrlcall(&trastinfo) < 0)
	{
		fprintf(stderr, "\n\aCan't read display contrast.\n") ;

		exit(1) ;
	}
	else
	{
		return (trastinfo . displayContrast) ;
	}
}

/*---------------------------------------------------------------------------*/

/* Function: settrast
   Purpose: Set the display contrast.
   Used by: Anyone.
   Returns: Nothing.
   Notes:
   Revision history:
	 1) Sunday December 7, 1997 -- 20:11:34.
*/

static void settrast(unsigned short trast)

{
	contrastinfo trastinfo ;


	(trastinfo . commandHeader . commandCode) = F_DISPCONTRASTSET ;
	(trastinfo . displayContrast) = trast ;
	if(sint10ctrlcall(&trastinfo) < 0)
	{
		fprintf(stderr, "\n\aCan't set display contrast.\n") ;

		exit(1) ;
	}
}

/*---------------------------------------------------------------------------*/

/* Function: getintensity
   Purpose: Read the current display backlight intensity setting.
   Used by: Anyone.
   Returns: Nothing.
   Notes:
   Revision history:
	 1) Sunday December 7, 1997 -- 20:11:34.
*/

static unsigned short getintensity(void)

{
	backlightintensityinfo tensityinfo ;


	(tensityinfo . commandHeader . commandCode) = F_DISPBKLTTENSITYGET ;
	if(sint10ctrlcall(&tensityinfo) < 0)
	{
		fprintf(stderr, "\n\aCan't read display backlight intensity.\n") ;

		exit(1) ;
	}
	else
	{
		return (tensityinfo . backlightIntensity) ;
	}
}

/*---------------------------------------------------------------------------*/

/* Function: settensity
   Purpose: Set the display backlight intensity.
   Used by: Anyone.
   Returns: Nothing.
   Notes:
   Revision history:
	 1) Sunday December 7, 1997 -- 20:11:34.
*/

static void settensity(unsigned short tensity)

{
	backlightintensityinfo tensityinfo ;


	(tensityinfo . commandHeader . commandCode) = F_DISPBKLTTENSITYSET ;
	(tensityinfo . backlightIntensity) = tensity ;
	if(sint10ctrlcall(&tensityinfo) < 0)
	{
		fprintf(stderr, "\n\aCan't set display backlight intensity.\n") ;

		exit(1) ;
	}
}

/*---------------------------------------------------------------------------*/

/* Function: gettimeout
   Purpose: Read the current display timeout.
   Used by: Anyone.
   Returns: Nothing.
   Notes:
   Revision history:
	 1) Sunday December 7, 1997 -- 20:11:34.
*/

static unsigned char gettimeout(void)

{
	displaytimeoutinfo timeoutinfo ;


	(timeoutinfo . commandHeader . commandCode) = F_DISPTIMEOUTGET ;
	if(sint10ctrlcall(&timeoutinfo) < 0)
	{
		fprintf(stderr, "\n\aCan't read display timeout.\n") ;

		exit(1) ;
	}
	else
	{
		return (timeoutinfo . timeOut) ;
	}
}

/*---------------------------------------------------------------------------*/

/* Function: settimeout
   Purpose: Set the display timeout.
   Used by: Anyone.
   Returns: Nothing.
   Notes:
   Revision history:
	 1) Sunday December 7, 1997 -- 20:11:34.
*/

static void settimeout(unsigned char timeout)

{
	displaytimeoutinfo timeoutinfo ;


	(timeoutinfo . commandHeader . commandCode) = F_DISPTIMEOUTSET ;
	(timeoutinfo . timeOut) = timeout ;
	if(sint10ctrlcall(&timeoutinfo) < 0)
	{
		fprintf(stderr, "\n\aCan't set display timeout.\n") ;

		exit(1) ;
	}
}

/*---------------------------------------------------------------------------*/

/* Function: adjuststuff
   Purpose: Make some display adjustments.
   Used by: Anyone.
   Returns: Nothing.
   Notes:
   Revision history:
	 1) Sunday December 7, 1997 -- 20:11:34.
*/

static void adjuststuff(void)

{
	unsigned char timeout ;

	unsigned short trast, newtrast, tensity, newtensity ;


	/* See if LCD video is up and running.
	*/
	check4lcddisplay() ;

	/* Display current display contrast and backlight settings.
	*/
	trast = gettrast() ;
	tensity = getintensity() ;
	timeout = gettimeout() ;
	printf("\nCurrent contrast:            %hu", trast) ;
	printf("\nCurrent backlight intensity: %hu", tensity) ;
	printf("\nCurrent display timeout:     %u minute(s)\n", timeout) ;

	/* Adjust display-related settings.
	*/
	newtensity = (tensity / 2) ;
	newtrast = (trast - (trast / 50)) ;

	timeout += 1 ; /* Make shut-off timeout one minute longer. (Will fail if timeout is already 60.) */
	settimeout(timeout) ;
	printf("\nBacklight timeout set to %u minutes.", timeout) ;

	printf("\nSetting backlight intensity to %hu...", newtensity) ;
	settensity(newtensity) ; /* Change the backlight intensity. */
	waitticks(73) ;
	printf("\nRestoring backlight intensity to %hu...", tensity) ;
	settensity(tensity) ; /* Restore the backlight intensity. */
	waitticks(73) ;

	printf("\nSetting display contrast to %u...", newtrast) ;
	settrast(newtrast) ; /* Change the display contrast. */
	waitticks(73) ;
	printf("\nRestoring display contrast to %hu...", trast) ;
	settrast(trast) ; /* Restore the display contrast. */
	waitticks(73) ;
}

/*---------------------------------------------------------------------------*/

/* Function: main
   Purpose: Program entry point.
   Used by: Program launcher.
   Returns: Never.
   Notes:
   Revision history:
	 1) Sunday December 7, 1997 -- 20:11:34.
*/

int main(unsigned argc, char *argv[])

{
	adjuststuff() ;

	printf("\n\nThat's all, folks.\n") ;

	exit(0) ;
}

/*---------------------------------------------------------------------------*/
