/*---------------------------------------------------------------------------*/
/*
   "SINT10.H"

   Include file containing rasterop constants and structures for use with
	 interrupt 0x10 functions.

   Version 1.8, Monday December 8, 1997 -- 17:23:27.

   Copyright (c) 1993 - 1997 by Mesa Electronics.
*/
/*---------------------------------------------------------------------------*/
/*
   Revision history.

   1) Version 1.0, Tuesday December 7, 1993 - 22:59:19.

      Code frozen for version 1.0.


   2) Version 1.1, Saturday January 8, 1994 - 23:39:44.

	  Changes from version 1.0:
	  1) Level-1 functions added to enum grfxfuncs (AH == F_SPCLVIDGRFX):
		 a) F_GRFXATTACHNTFONT uses the existing attachfontinfo structure to
			specify the font to be used as the non-TTY font.

		 b) F_GRFXDRAWCHAR uses the newly-added drawcharinfo structure to
			draw the specified character from the non-TTY font at the
			specified absolute pixel coordinate on the LCD screen.

		 c) F_GRFXSAVEREGNSIZEQ uses the newly-added dispregnsavesize
			structure to inquire how much memory is required to save the
			specified region of the LCD display.

		 d) F_GRFXSAVEREGN uses the newly-added dispregnsave structure to
			save the specified region of the LCD display to the specified
			memory buffer.

		 e) F_GRFXUNSAVEREGN uses the newly-added dispregnunsave structure to
			restore a previously-saved region of the LCD display.


   3) Version 1.2, Tuesday April 12, 1994 - 12:13:04.

	  Changes from version 1.1:
	  1) Level-2 functions added to enum grfxfuncs (AH == F_SPCLVIDGRFX):
		 a) F_GRFXSHOWPLANE switches the display to show only a specific
		    plane.


   4) Version 1.3, Sunday January 29, 1995 - 14:59:08.

      Changes from version 1.2:
	  1) bitbltinfo comment header updated.
	  2) unbitbltinfo structure added.


   5) Version 1.4, Monday February 6, 1995 - 13:21:07.

      Changes from version 1.3:
	  1) Level-3 functions added to enum grfxfuncs (AH == F_SPCLVIDGRFX):
		 a) F_GRFXUNBITBLTB is an alternate function number for
			F_GRFXUNBITBLT, but is classified as a level-3 function, so user
			code can use F_GRFXDISPINFOQ to check for its presence rather
			than having to run a test on F_GRFXUNBITBLT.

		 b) F_GRFXTTYCHARLOCQ translates a TTY region-relative rectangular
		    region location into absolute pixel coordinates.


   6) Version 1.5, Monday Monday July 31, 1995 - 00:52:16.

      Changes from version 1.4:
	  1) Level-4 functions added to enum grfxfuncs (AH == F_SPCLVIDGRFX):
		 a) F_GRFXFILLREGN uses the (new) fillrectinfo structure to fill a
		    rectangular region using the specified pattern/plane/rasterop.

		 b) F_GRFXTTYSTATEGET uses the (new) ttyregnstateinfo structure get
		    the current TTY region context information.

		 c) F_GRFXTTYSTATESET uses the ttyregnstateinfo structure to
		    set/modify/restore the TTY context.

	  2) Structures lcddisplaystateinfo added to support
		 dispcntrlfuncs.F_DISPSTATEGET.

	  3) Structure lcddisplaymodeinfo added to support
		 dispcntrlfuncs.F_DISPMODEGET and dispcntrlfuncs.F_DISPMODESET.

   7) Version 1.6, Thursday August 31, 1995 - 00:28:42.

      Changes from version 1.5:
	  1) Level-5 function added to enum grfxfuncs (AH == F_SPCLVIDGRFX):
		 a) F_GRFXFILLREGNX uses the (enhanced) fillrectinfo structure to
		    fill a rectangular region using the specified
			pattern/plane/rasterop.  F_GRFXFILLREGNX allows the pattern to
			have an initial rotation in X and Y specified.  This function
			obsoletes F_GRFXFILLREGN.


   8) Version 1.7, Wednesday August 14, 1996 -- 14:18:43.

      Changes from version 1.6:
	  1) Level-1 functions added to enum dispcntrlfuncs (AH ==
	     F_SPCLVIDCNTRL):
		 a) F_DISPBKLTTENSITYGET uses the backlightintensityinfo structure to
		    set the display backlight intensity.

		 b) F_DISPBKLTTENSITYGET uses the backlightintensityinfo structure to
		    get the current display backlight intensity setting.

	  2) Structure backlightintensityinfo added to support
	     dispcntrlfuncs.F_DISPBKLTTENSITYSET and
		 dispcntrlfuncs.F_DISPBKLTTENSITYGET.


   9) Version 1.8, Monday December 8, 1997 -- 17:23:27.

	  Changes from version 1.7:
	  1) Level-2 functions added to enum dispcntrlfuncs (AH ==
	     F_SPCLVIDCNTRL):
		 a) F_DISPTIMEOUTSET uses the displaytimeoutinfo structure to
		    set the display turn-off timeout.

		 b) F_DISPTIMEOUTGET uses the displaytimeoutinfo structure to
		    get the display turn-off timeout.

	  2) Structure displaytimeoutinfo added to support
	     dispcntrlfuncs.F_DISPTIMEOUTSET and dispcntrlfuncs.F_DISPTIMEOUTGET.
*/
/*---------------------------------------------------------------------------*/

#if !defined H_SINT10
#define H_SINT10

/*---------------------------------------------------------------------------*/

/* #defines.
*/
#define SPCLVIDFBASE    0x80 /* Base number for special video functions. */
#define M_RASTEROPBLINK	0x80 /* Blink flag for rasterops fields. */

/*---------------------------------------------------------------------------*/

/* Enumerations.
*/

/* Function numbers that are passed in register AH upon invocation of
     interrupt 0x10 in order to invoke Mesa-specific functions.
*/
typedef enum vidfuncsx
{	F_SPCLVIDCNTRL = SPCLVIDFBASE, /* Interrupt 0x10 control-related function selector. */
	F_SPCLVIDGRFX,                 /* Interrupt 0x10 graphics-related function selector. */

	MAXVIDFUNC
} vidfuncsx ;


/* Function numbers that are used to access the functions selected by
     vidfuncsx.F_SPCLVIDCNTRL.  (Control-related functions.)
*/
typedef enum dispcntrlfuncs
{	/* The following are code revision level 0 functions.
	*/
	F_DISPCNTRLINFOQ = 0, /* Get display control code revision level, etc. */
	F_DISPCHECKSETUP,     /* Validate setup parameters. */
	F_DISPRSVD0,          /* Reserved. */
	F_DISPINIT,           /* Activate the LCD display. */
	F_DISPMODEGET,        /* Get display operating modes information. */
	F_DISPMODESET,        /* Set display operating mode. */
	F_DISPRSVD1,          /* Reserved. */
	F_DISPRSVD2,
	F_DISPRSVD3,
	F_DISPRSVD4,
	F_DISPGETBUFPTR,      /* Get segment of display buffer. (Not currently implemented.) */
	F_DISPSETBUFPTR,      /* Set segment of display buffer. (Not currently implemented.) */
	F_DISPSTATEGET,       /* Get display on/off state information. */
	F_DISPSTATESET,       /* Set display on/off state. (Not currently implemented.) */
	F_DISPBKLTSTATEGET,   /* Get display backlight on/off state. */
	F_DISPBKLTSTATESET,   /* Set display backlight on/off state. */
	F_DISPCONTRASTGET,    /* Get current display contrast setting. */
	F_DISPCONTRASTSET,    /* Set display contrast. */

	/* The following are code revision level 1 functions.
	*/
	F_DISPBKLTTENSITYGET, /* Get current display backlight intensity setting. */
	F_DISPBKLTTENSITYSET, /* Set display backlight intensity. */

	/* The following are code revision level 2 functions.
	*/
	F_DISPTIMEOUTGET,     /* Get the display turn-off timeout. */
	F_DISPTIMEOUTSET,     /* Set the display turn-off timeout. */

	NUMDISPCNTRLFUNCS
} dispcntrlfuncs ;


/* Error codes returned by the functions selected by
     vidfuncsx.F_SPCLVIDCNTRL.  (Control-related functions.)
*/
typedef enum spcldisperrs
{	E_DISPNONE = 0, /* No error. */
	E_DISPBADFUNC,  /* Invalid subfunction number. */
	E_DISPBADPARM,  /* Invalid parameter. */

	NUMDISPCNTRLERRS
} spcldisperrs ;


/* Function numbers that are used to access the functions selected by
     vidfuncsx.F_SPCLVIDGRFX.  (Graphics-related functions.)
*/
typedef enum grfxfuncs
{	/* The following are code revision level 0 functions.  (Use F_GRFXDISPINFOQ
	     to obtain the code revision level.)
	*/
	F_GRFXDISPINFOQ = 0, /* Get graphics code revision level, etc. */
	F_GRFXDISPDIMQ,      /* Get display dimensions. */
	F_GRFXTTYINFOGET,    /* Get TTY region information. (Obsolete - see F_GRFXTTYSTATEGET.) */
	F_GRFXTTYINFOSET,    /* Set TTY region information. (Obsolete - see F_GRFXTTYSTATESET.) */
	F_GRFXCRSRXABLQ,     /* Get cursor xable state information. */
	F_GRFXCRSRXABL,      /* Xable cursor display. */
	F_GRFXTTYPATGET,     /* Get TTY region fill pattern. */
	F_GRFXTTYPATSET,     /* Set TTY region fill pattern. (Fills the TTY region when erased or scrolled.) */
	F_GRFXPATGET,        /* Get non-TTY region fill pattern. */
	F_GRFXPATSET,        /* Set non-TTY region fill pattern. */
	F_GRFXERASEREGN,     /* Erase a rectangular region. */
	F_GRFXDRAWDOT,       /* Draw a dot. */
	F_GRFXDRAWLINE,      /* Draw a single-pixel line. */
	F_GRFXBITBLT,        /* BitBLT to rectangular screen region. */
	F_GRFXUNBITBLT,      /* BitBLT from rectangular screen region. */
	F_GRFXERASETTY,      /* Erase the current TTY region. */
	F_GRFXERASENONTTY,   /* Erase the non-TTY region. */
	F_GRFXSETTTYDIM,     /* Set TTY region location/dimensions by point and size. */
	F_GRFXFONTINFOGET,   /* Get information about ROMBIOS-resident TTY fonts. */
	F_GRFXATTACHFONT,    /* Specify the font to be used as the TTY font. */
	F_GRFXSELFONT,       /* Select TTY font by number. */
	F_GRFXSETTTYLOC,     /* Set TTY region location/dimensions by rectangular region. */

	/* The following are code revision level 1 functions.
	*/
	F_GRFXTFONTINFOGET,  /* Get information about the current TTY font. */
	F_GRFXNTFONTINFOGET, /* Get information about the current non-TTY font. */
	F_GRFXATTACHNTFONT,  /* Specify the font to be the non-TTY font. */
	F_GRFXDRAWCHAR,      /* Draw a character from the non-TTY font at the specified pixel coordinate. */
	F_GRFXSAVEREGNSIZEQ, /* Get size of buffer required to save the specified LCD display region image. */
	F_GRFXSAVEREGN,      /* Save specified LCD display region image to buffer. */
	F_GRFXUNSAVEREGN,    /* Restore saved LCD display region from buffer to display. */

	/* The following are code revision level 2 functions.
	*/
	F_GRFXSHOWPLANE,     /* Display specific plane. */

	/* The following are code revision level 3 functions.
	*/
	F_GRFXUNBITBLTB,     /* Alternate access to F_GRFXUNBITBLT, which was unimplemented before level 3. */
	F_GRFXTTYCHARLOCQ,   /* Translate from TTY character region to absolute pixel coordinates. */

	/* The following are code revision level 4 functions.
	*/
	F_GRFXFILLREGN,      /* Fill a rectangular region with pattern/plane/rasterop. */
	F_GRFXTTYSTATEGET,   /* Get current TTY region state. */
	F_GRFXTTYSTATESET,   /* Set/modify/restore TTY region state. */

	/* The following are code revision level 5 functions.
	*/
	F_GRFXFILLREGNX,     /* Fill a rectangular region with pattern/plane/rasterop. The pattern may be shifted/rotated. */

	NUMGRFXFUNCS
} grfxfuncs ;


/* Error codes returned by the functions selected by vidfuncsx.F_SPCLVIDGRFX.
     (Graphics-related functions.)
*/
typedef enum spclgrfxerrs
{	E_GRFXNONE = 0, /* No error. */
	E_GRFXBADFUNC,  /* Invalid subfunction number. */
	E_GRFXBADPARM,  /* Invalid parameter. */
	E_GRFXBADCOORD, /* One or more invalid coordinates. */
	E_GRFXBADPLANE, /* Invalid plane. */

	NUMGRFXERRS
} spclgrfxerrs ;


/* Constants used for specifying the various rasterops.
*/
typedef enum rasterop
{	RASTEROP_REP = 0, /* Replace. */
	RASTEROP_NREP,    /* Invert and replace. */
	RASTEROP_AND,     /* AND. */
	RASTEROP_NAND,    /* Invert and AND. */
	RASTEROP_OR,	  /* OR. */
	RASTEROP_NOR,     /* Invert and OR. */
	RASTEROP_XOR,     /* XOR. */
	RASTEROP_NXOR,    /* Invert and XOR. */

	NUMRASTEROPS
} rasterop ;

/*---------------------------------------------------------------------------*/

/* Structures.
*/

/* Structure used to represent a point on the coordinate plane.

   Note that coordinates are numbered such that (0, 0) is the upper left-hand
     corner of the screen, and (XMax, YMax) is the lower right-hand corner of
     the screen.  Tough luck, purists.
*/
typedef struct point
{	unsigned short x ; /* X coordinate. */
	unsigned short y ; /* Y coordinate. */
} point ;


/* Structure used to represent a rectangular region on the coordinate plane.
*/
typedef struct rect
{	point minCoord ; /* Minimum coordinate. */
	point maxCoord ; /* Maximum coordinate. */
} rect ;


/* Structure used to pass a command to, and receive completion status from,
     vidfuncsx interrupt functions.  This structure is always the first
     element of any interrupt 010H special function command structure.

   When interrupt 0x10 is invoked, AH must be loaded with a function number
     defined by enumeration vidfuncsx, and CX:BX is loaded with a far pointer
     to one of the following structures that begins with an int10cmndhdr
     structure.  (CX contains the segment, BX contains the offset.)
*/
typedef struct int10cmndhdr
{	unsigned char commandCode ; /* The command code. */
	unsigned char errorCode ;   /* The error code. */
} int10cmndhdr ;


/* Structure used to inquire whether the special control functions code is
     available.  (Interrupt 0x10 register AH == F_SPCLVIDCNTRL,
     ctrlavailqinfo.commandHeader.commandCode == F_DISPCNTRLINFOQ.)
*/
typedef struct ctrlavailqinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	unsigned char toggleByte ;     /* Gets inverted if there's code to answer. */
	unsigned short revisionLevel ; /* Code revision level. */
} ctrlavailqinfo ;


/* Structure used to validate a display configuration inquiry.  (Interrupt
     0x10 register AH == F_SPCLVIDCNTRL,
     dispsetupchk.commandHeader.commandCode == F_DISPCHECKSETUP.)
*/
typedef struct dispsetupchk
{	int10cmndhdr commandHeader ; /* Command/status information. */

	unsigned short nybblesPerLine ; /* Number of nybbles per display line. (Input.) */
	unsigned short displayLines ;   /* Number of lines on the display. (Input.) */
	unsigned char numPlanes ;       /* Number of planes to use. (1 or 2.) */

	unsigned short paragraphsReq ; /* Number of paragraphs required for image buffer. (Output.) */
} dispsetupchk ;


/* Structure used to initialize the display.  (Interrupt 0x10 register AH ==
     F_SPCLVIDCNTRL, dispinit.commandHeader.commandCode == F_DISPINIT.)
*/
typedef struct dispinit
{	int10cmndhdr commandHeader ; /* Command/status information. */

	point displayDim ;              /* Display dimensions in pixels. */
	unsigned short nybblesPerLine ; /* Number of nybbles per display line. */
	unsigned short cl1Start ;       /* Starting nybble of CL1 from start of each line. */
	unsigned short cl1End ;         /* Ending nybble of CL1 from start of each line. */
	unsigned short flmStartOffset ; /* Nybble offset of start of FLM from start of display buffer. */
	unsigned short flmEndOffset ;   /* Nybble offset of last byte of FLM from start of display buffer. */

	unsigned char numPlanes ; /* Number of planes to use. (1 or 2.) */

	/* The size of the buffer required to hold the entire LCD display image must
	     be determined by invoking interrupt 0x10 function number F_SPCLVIDCNTRL,
	     subfunction F_DISPCHECKSETUP (which uses structure dispsetupchk) and
	     examining the number returned in dispsetupchk.paragraphsReq.
	     Multiplying that number by 16 yields the number of bytes required to
	     contain the LCD display buffer.  The display buffer referenced by
	     .baseParagraph must be paragraph aligned in system memory.
	*/
	unsigned short baseParagraph ; /* Base paragraph of video buffer to be used. */

	unsigned char invertDisp ; /* Invert the display. */
} dispinit ;


/* Structure used to get and set the LCD display mode.  (Interrupt 0x10
     register AH == F_SPCLVIDCNTRL,
     lcddisplaymodeinfo.commandHeader.commandCode == F_DISPMODEGET or
     F_DISPMODESET.)
*/
typedef struct lcddisplaymodeinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	unsigned char reverseVideo ; /* Reverse video flag. */
} lcddisplaymodeinfo ;


/* Structure used to get and set the LCD display contrast.  (Interrupt 0x10
     register AH == F_SPCLVIDCNTRL, contrastinfo.commandHeader.commandCode ==
     F_DISPCONTRASTGET or F_DISPCONTRASTSET.)
*/
typedef struct contrastinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	/* The display contrast value has a full scale value of 65535 and a minimum
	     value of 0.  A contrast setting of 32767 therefore gives a setting of
	     about 50 percent full scale.
	*/
	unsigned short displayContrast ; /* The contrast value. */
} contrastinfo ;


/* Structure used to get and set the LCD display on/off state.  (Interrupt
     0x10 register AH == F_SPCLVIDCNTRL,
     lcddisplaystateinfo.commandHeader.commandCode == F_DISPSTATEGET or
     F_DISPSTATESET.)
*/
typedef struct lcddisplaystateinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	unsigned char displayOn ; /* Current/desired display on/off state. */
} lcddisplaystateinfo ;


/* Structure used to get and set the LCD display backlight state.  (Interrupt
     0x10 register AH == F_SPCLVIDCNTRL,
     backlitestateinfo.commandHeader.commandCode == F_DISPBKLTSTATEGET or
     F_DISPBKLTSTATESET.)
*/
typedef struct backlitestateinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	unsigned char backliteOn ; /* Non-zero for on. */
} backlitestateinfo ;


/* Structure used to inquire whether or not the display is active and
     available.  (Interrupt 0x10 register AH == F_SPCLVIDGRFX,
     dispavailqinfo.commandHeader.commandCode == F_GRFXDISPINFOQ.)

   Note that attempting to use any functions that access the display or
     display hardware will probably result in undefined (i.e. system
     crash-like) behavior if an invocation of function F_GRFXDISPINFOQ would
     return with .isAvailable set to 0.  Before attempting to access the
     display, interrupt 0x10 function number F_SPCLVIDCNTRL, subfunction
     F_DISPINIT (which uses structure dispinit) should be called to ensure
     that the LCD display is on-line.
*/
typedef struct dispavailqinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	unsigned char toggleByte ;     /* Gets inverted if there's code to answer. */
	unsigned char isAvailable ;    /* Availability flag - non-zero if the display is active. */
	unsigned short revisionLevel ; /* Code revision level. */
} dispavailqinfo ;


/* Structure used to inquire about graphic display dimensions.  (Interrupt
     0x10 register AH == F_SPCLVIDGRFX,
     dispdimqinfo.commandHeader.commandCode == F_GRFXDISPDIMQ.)
*/
typedef struct dispdimqinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	point dispDim ;           /* Maximum X, Y coordinates. */
	unsigned char numPlanes ; /* Number of display planes. (Blink uses 2 planes.) */
} dispdimqinfo ;


/* Structure used to get and set TTY region information.  (Interrupt 0x10
     register AH == F_SPCLVIDGRFX, ttyregninfo.commandHeader.commandCode ==
     F_GRFXTTYINFOGET or F_GRFXTTYINFOSET.)

   Note that this function/structure is obsolete; use ttystateinfo with
     F_GRFXTTYSTATEGET and F_GRFXTTYSTATESET.
*/
typedef struct ttyregninfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	point ttyUpperLeft ; /* Upper left corner of the TTY region on the display. */
	point ttyDim ;       /* TTY region dimensions. */

	/* The following information can refer to a user-supplied font somewhere in
	     system memory, or it can refer to one of the ROMBIOS-resident TTY fonts.
	     Information about ROMBIOS-resident TTY fonts can be obtained by invoking
	     interrupt 0x10 function number F_SPCLVIDCNTRL, subfunction
	     F_GRFXFONTINFO (which uses structure ttyfontinfo).
	*/
	unsigned char charWidth ;    /* Width of a TTY character cell in pixels. */
	unsigned char charHeight ;   /* Height of a TTY character cell in pixels. */
	unsigned char bytesPerLine ; /* Number of bytes in a TTY character raster line. */
	unsigned char bytesPerChar ; /* Number of bytes per TTY character. */
	unsigned short charsInFont ; /* Number of characters in the TTY font. Must be at least 255 for TTY use. */
	unsigned char far *fontPtr ; /* Pointer to the monospaced font to be used for TTY region. */
} ttyregninfo ;


/* Structure used to get and set the TTY and graphics region fill patterns.
     (Interrupt 0x10 register AH == F_SPCLVIDGRFX,
     fillpatinfo.commandHeader.commandCode == F_GRFXTTYPATGET,
     F_GRFXTTYPATSET, F_GRFXPATGET, or F_GRFXPATSET.)
*/
typedef struct fillpatinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	unsigned char far *patPtr ; /* Pointer to the 8-byte pattern buffer. */
} fillpatinfo ;


/* Structure used to erase a rectangular region of the display.  (Interrupt
     0x10 register AH == F_SPCLVIDGRFX, eraseinfo.commandHeader.commandCode
     == F_GRFXERASEREGN.)  The erased region is filled with the fill pattern
     set by interrupt 0x10 function F_SPCLVIDGRFX, subfunction F_GRFXPATSET
     (which uses structure fillpatinfo).
*/
typedef struct eraserectinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	rect theRegion ; /* The region to be erased. */
} eraserectinfo ;


/* Structure used to inquire about and set the cursor display xable state.
     (Interrupt 0x10 register AH == F_SPCLVIDGRFX,
     crsrxableinfo.commandHeader.commandCode == F_GRFXCRSRXABLQ or
     F_GRFXCRSRXABL.)  Setting the cursor xable state also returns the
     current setting of the cursor xable state in .cursorEnable.
*/
typedef struct crsrxableinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	unsigned char cursorEnable ; /* Cursor state flag. */
} crsrxableinfo ;


/* Structure used to draw a single dot on the display.  (Interrupt 0x10
     register AH == F_SPCLVIDGRFX, drawdotinfo.commandHeader.commandCode ==
     F_GRFXDRAWDOT.)

   .planeMask contains the mask of plane to be drawn to.  Setting bit 0 draws
     to plane 0, setting bit 1 draws to plane 1, etc.  If no bits are set,
     drawing is performed on all planes.  If a non-existent plane is
     specified, an error is reported.

   Using M_RASTEROPBLINK to set the blink bit of .rasterOp will cause blink,
     but only if more than one display plane is available and .planeMask is
     set to 0x00 or specifies all planes.  The blink bit of .rasterOp is
     ignored if only one display plane is available or .planeMask specifies
     other than all planes.
*/
typedef struct drawdotinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	point thePoint ;         /* One point. */
	unsigned char dotColor ; /* The dot "color". (Non-zero for on, zero for off.) */

	unsigned char rasterOp ;  /* The rasterop (from enum rasterop) to be used. */
	unsigned char planeMask ; /* Drawing plane mask. */
} drawdotinfo ;


/* Structure used to draw a single dot-width line on the display.  (Interrupt
     0x10 register AH == F_SPCLVIDGRFX,
     drawline1info.commandHeader.commandCode == F_GRFXDRAWLINE.)

   .planeMask contains the mask of plane to be drawn to.  Setting bit 0 draws
     to plane 0, setting bit 1 draws to plane 1, etc.  If no bits are set,
     drawing is performed on all planes.  If a non-existent plane is
     specified, an error is reported.

   Using M_RASTEROPBLINK to set the blink bit of .rasterOp will cause blink,
     but only if more than one display plane is available and .planeMask is
     set to 0x00 or specifies all planes.  The blink bit of .rasterOp is
     ignored if only one display plane is available or .planeMask specifies
     other than all planes.
*/
typedef struct drawline1info
{	int10cmndhdr commandHeader ; /* Command/status information. */

	point startPoint ;        /* One point. */
	point endPoint ;          /* The other point. */
	unsigned char lineColor ; /* The line "color". (Non-zero for on, zero for off.) */

	unsigned char skipFirstDot ; /* Non-zero to skip drawing first dot. */
	unsigned char skipLastDot ;  /* Non-zero to skip srawing last dot. */

	unsigned char rasterOp ;  /* The rasterop (from enum rasterop) to be used. */
	unsigned char planeMask ; /* Drawing plane mask. */
} drawline1info ;


/* Structure used to bitBLT information from system memory to the LCD display
     screen.  (Interrupt 0x10 register AH == F_SPCLVIDGRFX,
     bitbltinfo.commandHeader.commandCode == F_GRFXBITBLT.)

   .planeMask contains a mask that selects the LCD display destination plane.
     Setting bit 0 draws to plane 0, setting bit 1 draws to plane 1, etc.  If
     no bits are set, drawing is performed on all planes.  If a non-existent
     plane is specified, an error is reported.

   Using M_RASTEROPBLINK to set the blink bit of .rasterOp will cause the
     resulting image to blink, but only if more than one display plane is
     available and .planeMask is set to 0x00 or specifies all planes.  The
     blink bit of .rasterOp is ignored if only one display plane is available
     or .planeMask specifies other than all planes.

   The following members must be initalized: .sourceRect, .sourcePitch,
     .sourcePtr, .destPoint, .rasterOp, .planeMask.  .destPitch and .destPtr
     are obsolete and exist only to maintain structure member alignment.
*/
typedef struct bitbltinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	rect sourceRect ;              /* The source region. */
	unsigned short sourcePitch ;   /* Number of bytes per raster line in source. */
	unsigned char far *sourcePtr ; /* Pointer to source image. */

	point destPoint ;            /* Upper left corner of destination region. */
	unsigned short destPitch ;   /* (Obsolete.) */
	unsigned char far *destPtr ; /* (Obsolete.) */

	unsigned char rasterOp ;  /* The rasterop (from enum rasterop) to be used for the BLT. */
	unsigned char planeMask ; /* Drawing plane mask. */
} bitbltinfo ;


/* Structure used to bitBLT information from the LCD display screen to system
     memory.  (Interrupt 0x10 register AH == F_SPCLVIDGRFX,
     unbitbltinfo.commandHeader.commandCode == F_GRFXUNBITBLT or
     F_GRFXUNBITBLTB.)

   .planeMask contains the mask of the LCD display source plane.  Setting bit
     0 reads from plane 0, setting bit 1 reads from plane 1, etc.  If no bits
     are set, more that one bit is set, or if a non-existent plane is
     specified, an error is reported.

   For F_GRFXUNBITBLT, the M_RASTEROPBLINK bit of .rasterOp is ignored.
*/
typedef struct unbitbltinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	rect destRect ;              /* The destination region. */
	unsigned short destPitch ;   /* Number of bytes per raster line in destination. */
	unsigned char far *destPtr ; /* Pointer to destination image. */

	point sourcePoint ; /* Upper left corner of source region. */

	unsigned char rasterOp ;  /* The rasterop (from enum rasterop) to be used for the BLT. */
	unsigned char planeMask ; /* Drawing plane mask. */
} unbitbltinfo ;


/* Structure used to erase the TTY or non-TTY regions of the screen.
     (Interrupt 0x10 register AH == F_SPCLVIDGRFX,
     erasedispinfo.commandHeader.commandCode == F_GRFXERASETTY or
     F_GRFXERASENONTTY.)
*/
typedef struct erasedispinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */
} erasedispinfo ;


/* Structure used to set the dimensions and location of the TTY region only.
     (Interrupt 0x10 register AH == F_SPCLVIDGRFX,
     erasedispinfo.commandHeader.commandCode == F_GRFXSETTTYDIM.)
*/
typedef struct ttydiminfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	point ttyUpperLeft ; /* Upper left corner of the TTY region on the display. */
	point ttyDim ;       /* TTY region dimensions. */
} ttydiminfo ;


/* Structure used to obtain information about the ROMBIOS-resident TTY fonts.
     (Interrupt 0x10 register AH == F_SPCLVIDGRFX,
     ttyfontinfo.commandHeader.commandCode == F_GRFXFONTINFOGET.)

   The procedure for enumerating the TTY fonts is to initialize .fontNumber
     to 0, and call interrupt 0x10 with AH == F_SPCLVIDGRFX and CX:BX
     pointing to a ttyfontinfo structure.  After each successful call to
     interrupt 0x10, .fontNumber is incremented and interrupt 0x10 is called
     again.  When .commandHeader.errorCode == E_GRFXBADPARM, information for
     each of the available ROMBIOS-resident fonts has been returned and
     .fontNumber contains the number of ROMBIOS-resident fonts.
*/
typedef struct ttyfontinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	unsigned char fontNumber ; /* Number of font for which information is to be returned. */

	unsigned char charWidth ;    /* Returned width of a TTY character cell in pixels. */
	unsigned char charHeight ;   /* Returned height of a TTY character cell in pixels. */
	unsigned char bytesPerLine ; /* Returned number of bytes in a TTY character raster line. */
	unsigned char bytesPerChar ; /* Returned number of bytes per TTY character. */
	unsigned short charsInFont ; /* Returned number of characters in the TTY font. */
	unsigned char far *fontPtr ; /* Returned pointer to the monospaced TTY font. */
} ttyfontinfo ;


/* Structure used to specify or inquire about the particulars of a TTY or
     non-TTY font.  (Interrupt 0x10 register AH == F_SPCLVIDGRFX,
     attachfontinfo.commandHeader.commandCode == F_GRFXATTACHFONT,
     F_GRFXATTACHNTFONT, F_GRFXTFONTINFOGET, or F_GRFXNTFONTINFOGET.)

   Note that a font being specified for TTY use (F_GRFXATTACHFONT) must
     contain at least 256 character images (i.e. the entire IBM extended
     ASCII character set).  A font being specified for non-TTY use
     (F_GRFXATTACHNTFONT) can contain any non-zero number of character
     images, provided the entire font image is no larger than 65520 bytes.
*/
typedef struct attachfontinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	unsigned char charWidth ;    /* Width of a character cell in pixels. */
	unsigned char charHeight ;   /* Height of a character cell in pixels. */
	unsigned char bytesPerLine ; /* Number of bytes in a character raster line. */
	unsigned char bytesPerChar ; /* Number of bytes per character. */
	unsigned short charsInFont ; /* Number of characters in the font. */
	unsigned char far *fontPtr ; /* Pointer to the monospaced font. */
} attachfontinfo ;


/* Structure used to select a ROMBIOS-resident TTY font by its number.
     (Interrupt 0x10 register AH == F_SPCLVIDGRFX,
     selfontinfo.commandHeader.commandCode == F_GRFXSELFONT.)

     When .commandHeader.errorCode == E_DISPBADPARM, .fontNumber contains the
       number of ROMBIOS-resident fonts.
*/
typedef struct selfontinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	unsigned char fontNumber ; /* Number of font for which information is to be returned. */
} selfontinfo ;


/* Structure used to set the dimensions and location of the TTY region only.
     (Interrupt 0x10 register AH == F_SPCLVIDGRFX,
     ttylocinfo.commandHeader.commandCode == F_GRFXSETTTYLOC.)
*/
typedef struct ttylocinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	rect ttyRegion ; /* The rectangular region of the display to be used as the TTY region. */
} ttylocinfo ;


/* Structure used to draw a non-TTY character at the specified pixel
     coordinate.  (Interrupt 0x10 register AH == F_SPCLVIDGRFX,
     drawcharinfo.commandHeader.commandCode == F_GRFXDRAWCHAR.)

   .planeMask contains the mask of plane to be drawn to.  Setting bit 0 draws
     to plane 0, setting bit 1 draws to plane 1, etc.  If no bits are set,
     drawing is performed on all planes.  If a non-existent plane is
     specified, an error is reported.

   Using M_RASTEROPBLINK to set the blink bit of .rasterOp will cause blink,
     but only if more than one display plane is available and .planeMask is
     set to 0x00 or specifies all planes.  The blink bit of .rasterOp is
     ignored if only one display plane is available or .planeMask specifies
     other than all planes.

   If an invalid character number is specified for the current non-TTY font,
     the first character of that font is used as a slug.
*/
typedef struct drawcharinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	point destPoint ;        /* Upper left corner coordinate of character cell. */
	unsigned short theChar ; /* Number of the character from the non-TTY font. */

	unsigned char rasterOp ;  /* The rasterop (from enum rasterop) to be used. */
	unsigned char planeMask ; /* Drawing plane mask. */
} drawcharinfo ;


/* Structure used to inquire how large a memory buffer is required to save
     the specified region of the LCD display.  (Interrupt 0x10 register AH ==
     F_SPCLVIDGRFX, dispregnsavesize.commandHeader.commandCode ==
     F_GRFXSAVEREGNSIZEQ.)
*/
typedef struct dispregnsavesize
{	int10cmndhdr commandHeader ; /* Command/status information. */

	rect theRegion ;                /* The region of interest. */
	unsigned long regnSaveBufSize ; /* Number of bytes in the region to be saved. */
} dispregnsavesize ;


/* Structure used to save the specified region of the LCD display to a memory
     buffer.  (Interrupt 0x10 register AH == F_SPCLVIDGRFX,
     dispregnsave.commandHeader.commandCode == F_GRFXSAVEREGN.)
*/
typedef struct dispregnsave
{	int10cmndhdr commandHeader ; /* Command/status information. */

	rect theRegion ;                 /* The region to be saved. */
	unsigned char far *regnSavePtr ; /* Pointer to the buffer in which the display region is to be saved. */
} dispregnsave ;


/* Structure used to restore a region of the LCD display from a memory
     buffer.  (Interrupt 0x10 register AH == F_SPCLVIDGRFX,
     dispregnunsave.commandHeader.commandCode == F_GRFXUNSAVEREGN.)
*/
typedef struct dispregnunsave
{	int10cmndhdr commandHeader ; /* Command/status information. */

	unsigned char far *regnSavePtr ; /* Pointer to the buffer in which the display image has been saved. */
} dispregnunsave ;


/* Structure used to display a specific video plane.  (Interrupt 0x10
     register AH == F_SPCLVIDGRFX, showplaneinfo.commandHeader.commandCode ==
     F_GRFXSHOWPLANE.)
*/
typedef struct showplaneinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	signed char planeNum ; /* Specific plane number to be displayed. -1 to return to blink mode. */
} showplaneinfo ;


/* Structure used to translate from TTY character region to absolute pixel
     coordinates.  (Interrupt 0x10 register AH == F_SPCLVIDGRFX,
     xlatettyregion.commandHeader.commandCode == F_GRFXTTYCHARLOCQ.)
*/
typedef struct xlatettyregion
{	int10cmndhdr commandHeader ; /* Command/status information. */

	rect theRegion ; /* The region. Character coordinates input, pixel coordinates output. */
} xlatettyregion ;


/* Structure used to fill an arbitrary display region in the specified
     plane(s) with the specified pattern.  (Interrupt 0x10 register AH ==
     F_SPCLVIDGRFX, fillrectinfo.commandHeader.commandCode ==
     F_GRFXFILLREGN (obsolete) or F_GRFXFILLREGNX.)

   Using M_RASTEROPBLINK to set the blink bit of .rasterOp will cause blink,
     but only if more than one display plane is available and .planeMask is
     set to 0x00 or specifies all planes.  The blink bit of .rasterOp is
     ignored if only one display plane is available or .planeMask specifies
     other than all planes.
*/
typedef struct fillrectinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	rect theRegion ;            /* The region to be erased. */
	unsigned char far *patPtr ; /* Pointer to the 8-byte pattern buffer. */
	unsigned char rasterOp ;    /* The rasterop (from enum rasterop) to be used. */
	unsigned char planeMask ;   /* Drawing plane mask. */

	/* With F_GRFXFILLREGNX, .patPoint controls the rotation of the pattern
		 in X and Y if .regionRelative is non-zero.  In such a case,
		 .patPoint specifies the pattern pixel that is to be applied to the
		 upper left corner of the filled region.  If .regionRelative is zero,
		 the pattern is applied the usual way - aligned with absolute display
		 coordinate (0,0).  .regionRelative and .patPoint are ignored by
		 F_GRFXFILLREGN.
	*/
	unsigned char regionRelative ; /* Non-zero if pattern application is region-relative, zero for display origin-relative. */
	point patPoint ;               /* The coordinate within the pattern to appear at the origin of the filled region. */
} fillrectinfo ;


/* Structure used to get and set TTY region state information.  (Interrupt
     0x10 register AH == F_SPCLVIDGRFX,
     ttyregnstateinfo.commandHeader.commandCode == F_GRFXTTYSTATEGET or
     F_GRFXTTYSTATESET.)
*/
typedef struct ttyregnstateinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	point ttyUpperLeft ; /* Upper left corner of the TTY region on the display. */
	point ttyDim ;       /* TTY region dimensions. */

	/* The following information can refer to a user-supplied font somewhere in
	     system memory, or it can refer to one of the ROMBIOS-resident TTY fonts.
	     Information about ROMBIOS-resident TTY fonts can be obtained by invoking
	     interrupt 0x10 function number F_SPCLVIDCNTRL, subfunction
	     F_GRFXFONTINFO (which uses structure ttyfontinfo).
	*/
	unsigned char charWidth ;    /* Width of a TTY character cell in pixels. */
	unsigned char charHeight ;   /* Height of a TTY character cell in pixels. */
	unsigned char bytesPerLine ; /* Number of bytes in a TTY character raster line. */
	unsigned char bytesPerChar ; /* Number of bytes per TTY character. */
	unsigned short charsInFont ; /* Number of characters in the TTY font. Must be at least 255 for TTY use. */
	unsigned char far *fontPtr ; /* Pointer to the monospaced font to be used for TTY region. */

	unsigned char cursorDimX ;    /* Width of cursor image in pixels. */
	unsigned char cursorDimY ;    /* Height of cursor image in pixels. */
	unsigned char cursorOffsetX ; /* Start pixel of cursor image relative to left edge of character cell. */
	unsigned char cursorOffsetY ; /* Start pixel of cursor image relative to top edge of character cell. */
	point cursorLoc ; /* The location of the cursor within the TTY region. */

	unsigned char eraseRegion ; /* Used with F_GRFXTTYSTATESET - non-zero if the TTY region is to be erased. */
} ttyregnstateinfo ;


/* Structure used to get and set the display backlight intensity.  (Interrupt
     0x10 register AH == F_SPCLVIDCNTRL,
	 backlightintensityinfo.commandHeader.commandCode == F_DISPBKLTTENSITYGET
	 or F_DISPBKLTTENSITYSET.)
*/
typedef struct backlightintensityinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	/* The backlight intensity value has a full scale value of 65535 and a
	     minimum value of 0.  A setting of 32767 therefore represents about 50
		 percent full scale.
	*/
	unsigned short backlightIntensity ; /* The intensity value. */
} backlightintensityinfo ;


/* Structure used to get and set the display turn-off timeout.  (Interrupt
     0x10 register AH == F_SPCLVIDCNTRL,
	 backlightintensityinfo.commandHeader.commandCode == F_DISPTIMEOUTGET or
	 F_DISPTIMEOUTSET.)
*/
typedef struct displaytimeoutinfo
{	int10cmndhdr commandHeader ; /* Command/status information. */

	unsigned char timeOut ; /* Backlight timeout in minutes. (1...60.) */
} displaytimeoutinfo ;

/*---------------------------------------------------------------------------*/

#endif
