/*---------------------------------------------------------------------------*/
/*
   SCANDEMO.C"

   Demonstration program for the Mesa Electronics FPTX membrane keyboard.

   Version 1.0, Saturday September 27, 1997 -- 17:23:59.
*/
/*---------------------------------------------------------------------------*/
/*
   Compiler: Borland C++, version 3.1.
*/
/*---------------------------------------------------------------------------*/
/*
   Revision history.

   1) Version 1.0, Friday July 18, 1997 -- 18:26:41.

      Code frozen for version 1.0.
*/
/*---------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <dos.h>
#include "biosid.h"
#include "pcpusrvc.h"

/*---------------------------------------------------------------------------*/

/* #defines.
*/
#define ERRORBEEPDIVISOR 4000U /* Pitch for FIFO-full beeps. */
#define KEYBREAKDIVISOR  600U /* Pitch for key break events. */

#define KBSEG 0x0040 /* Keyboard FIFO segment. */

/*---------------------------------------------------------------------------*/

/* Set stack and heap length to known values.
*/
extern unsigned _heaplen = 256 ;
extern unsigned _stklen = 1024 ;

/*---------------------------------------------------------------------------*/

/* Local external variables.
*/

/* Far pointers to keyboard FIFO control variables.
*/
static unsigned short far *KBBufferGetPtrAdd = MK_FP(KBSEG, 0x001A) ;
static unsigned short far *KBBufferPutPtrAdd = MK_FP(KBSEG, 0x001C) ;
static unsigned short far *KBBufStartOffstAdd = MK_FP(KBSEG, 0x0080) ;
static unsigned short far *KBBufEndOffstAdd = MK_FP(KBSEG, 0x0082) ;

/* Membrane keyboard key to IBM keycode translation table.
*/
static unsigned short MembraneKeyXlateTable[] = \
{
	0x1E61, /* 'a'. */
	0x3062, /* 'b'. */
	0x2E63, /* 'c'... */
	0x2064,
	0x1265,
	0x2166,
	0x2267,
	0x2368,
	0x1769,
	0x246A,
	0x256B,
	0x266C,
	0x326D,
	0x316E,
	0x1C0D,	/* <Enter>. */
	0x0E08  /* <Backspace>. */
} ;

/*---------------------------------------------------------------------------*/

/* Function: pubservicecall
   Purpose: Call our public services code.
   Used by: Anyone.
   Returns: Pass/fail status.
   Notes:
   Revision history:
     1) Saturday July 12, 1997 -- 12:20:16.
*/

static signed pubservicecall(void far *parmptr)

{
	static unsigned char far *servicevec = MK_FP(ROMBIOSIDSEG, (ROMBIOSIDOFF + offsetof(rombiosid, jmp2PublicServices))) ;


	asm {
				mov		bx,word ptr parmptr[0]	// Offset of far pointer.
				mov		cx,word ptr parmptr[2]	// Segment of far pointer.
				call	dword ptr [servicevec]	// Call the CPU public services code.
				cld								// (Just in case.)
	}
	return (((((pfuncshdr far *)parmptr) -> errorCode) != E_PUBSRVCERRNONE) ? -1 : 0) ;
}

/*---------------------------------------------------------------------------*/

/* Function: putkeycodeinkeyboardfifo
   Purpose: Insert a keycode in the system keyboard FIFO if there's room.
   Used by: newscanhandler().
   Returns: Pass/fail status.
   Notes:
   Revision history:
     1) Saturday July 12, 1997 -- 12:20:16.
*/

static signed putkeycodeinkeyboardfifo(unsigned short keycode)

{
	unsigned short newputptr, putptr ;


	newputptr = putptr = *KBBufferPutPtrAdd ;
	newputptr += sizeof(unsigned short) ;
	if(newputptr >= *KBBufEndOffstAdd)
	{
		newputptr = *KBBufStartOffstAdd ;
	}
	if(newputptr != *KBBufferGetPtrAdd)
	{	/* There's room in the FIFO.
		*/
		*(unsigned short far *)MK_FP(KBSEG, putptr) = keycode ;
		*KBBufferPutPtrAdd = newputptr ;

		return 0 ;
	}
	else
	{	/* The FIFO is full.
		*/
		return -1 ;
	}
}

/*---------------------------------------------------------------------------*/

/* Function: newscanhandler
   Purpose: Demo membrane keyboard scan event handler.  Approximates the
	 behavior of the ROMBIOS-resident membrane keyboard event handler.
   Used by: ROMBIOS-resident membrane keyboard scan code.
   Returns: Control masks in AX, beep duration value in BX.
   Notes:
     -Upon entry, AX contains the number of the key that generated the event,
		and BX contains the event code, one of: KSTATE_MAKE (a key has been
		pressed), KSTATE_REPEAT (a key is being held down), or KSTATE_BREAK
		(a key has been released).
	 -All registers other than AX and BX must be preserved.
     -Upon return to the ROMBIOS, the following sequence of operations occur:

		if(AX & M_MEMBBAKLITEONREQ)
		{	if(Backlight is enabled by system configuration)
			{	Turn on backlight.
				Begin backlight shutoff timeout.
			}
			else
			{	Don't touch backlight or backlight shutoff timer.
			}
		}
		if(AX & M_MEMBBEEPONREQ)
		{	Turn on system beep.
			if(BX == 0)
			{	BX = System beep duration.
			}
		}
		if(BX != 0)
		{	System beep shutoff timer = BX.
			Begin system beep shutoff timeout.
		}
		else
		{	Don't touch system beep shutoff timer.
		}

   Revision history:
     1) Saturday July 12, 1997 -- 12:20:16.
	 2) Friday July 18, 1997 -- 18:26:41.
	    -Handling added for backlight turn-on.
*/

static void interrupt newscanhandler(unsigned bp, unsigned di, unsigned si, unsigned ds, unsigned es, unsigned dx, unsigned cx,
	unsigned bx, unsigned ax)

{
	switch(bx)
	{
		case KSTATE_MAKE: /* Key make event. */
		case KSTATE_REPEAT: /* Key repeat event. */
			if(putkeycodeinkeyboardfifo(MembraneKeyXlateTable[ax]) >= 0)
			{	/* There was room for the character in the keyboard FIFO.
				*/
				ax = (M_MEMBBEEPONREQ | M_MEMBBAKLITEONREQ) ; /* Use beep pitch set by pubmembranekbinfo.clickDivisor. */
				bx = 0 ; /* Use the beep duration defined by pubmembranekbinfo.clickScans. */
			}
			else
			{	/* There's no room in the FIFO; start the error beep.  (The ROMBIOS will
					 turn off the beep for us.)
				*/
				outportb(0x43, 0xB6) ;
				outportb(0x42, (unsigned char)ERRORBEEPDIVISOR) ;
				outportb(0x42, (unsigned char)(ERRORBEEPDIVISOR >> 8)) ;
				outportb(0x61, (inportb(0x61) | 0x03)) ;
				ax = M_MEMBBAKLITEONREQ ; /* Turn on backlight, don't use system beep pitch. */
				bx = 20 ; /* Error beep duration overrides pubmembranekbinfo.ovfBeepScans. */
			}

			break ;

		case KSTATE_BREAK: /* Key break event. */
			/* Note that the default (ROMBIOS-resident) membrane keyboard event handler
			     doesn't beep on KSTATE_BREAK events - this is just for demo purposes.
			*/
			outportb(0x43, 0xB6) ;
			outportb(0x42, (unsigned char)KEYBREAKDIVISOR) ;
			outportb(0x42, (unsigned char)(KEYBREAKDIVISOR >> 8)) ;
			outportb(0x61, (inportb(0x61) | 0x03)) ;
			ax = M_MEMBBAKLITEONREQ ; /* Turn on backlight, don't use system beep pitch. */
			bx = 2 ; /* Beep duration. */

			break ;
	}
}

/*---------------------------------------------------------------------------*/

/* Function: membenabledq
   Purpose: Return the membrane keyboard xable status.
   Used by: Anyone.
   Returns: Nothing.
   Notes:
   Revision history:
     1) Saturday July 12, 1997 -- 12:20:16.
*/

static int membenabledq(void)

{
	pubmembranekbxable info ;


	(info . commandHeader . commandCode) = F_PUBSRVCMEMBKBXABLEQ ;
	if(pubservicecall(&info) < 0)
	{
		fprintf(stderr, "\n\aCan't get membrane keyboard xable status.\n") ;

		exit(1) ;
	}
	return !!(info . scanEnable) ;
}

/*---------------------------------------------------------------------------*/

/* Function: displaymembraneinfo
   Purpose: Display the current membrane keyboard control information.
   Used by: Anyone.
   Returns: Nothing.
   Notes:
   Revision history:
     1) Saturday July 12, 1997 -- 12:20:16.
*/

static void displaymembraneinfo(void)

{
	pubmembranekbinfo scaninfo ;


	/* Display current membrane keyboard scan control settings.
	*/
	(scaninfo . commandHeader . commandCode) = F_PUBSRVCMEMBSCANINFOQ ;
	if(pubservicecall(&scaninfo) < 0)
	{
		fprintf(stderr, "\n\aCan't read membrane keyboard scan control information.\n") ;

		exit(1) ;
	}

	printf("\n\nCurrent membrane keyboad particulars:") ;
	printf("\n  Number of keys:         %hu", (scaninfo . numKeys)) ;
	printf("\n  Click:                  %s", ((scaninfo . clickEnable) ? "Enabled" : "Disabled")) ;
	printf("\n  Click scans:            %hu", (scaninfo . clickScans)) ;
	printf("\n  Click beep divisor:     %hu", (scaninfo . clickDivisor)) ;
	printf("\n  Overflow beep:          %s", ((scaninfo . ovfBeepEnable) ? "Enabled" : "Disabled")) ;
	printf("\n  Overflow beep scans:    %hu", (scaninfo . ovfBeepScans)) ;
	printf("\n  Overflow beep divisor:  %hu", (scaninfo . ovfBeepDivisor)) ;
	printf("\n  Debounce scans:         %hu", (scaninfo . debounceScans)) ;
	printf("\n  Repeat rate scans:      %hu", (scaninfo . repeatRateScans)) ;
	printf("\n  Repeat delay scans:     %hu", (scaninfo . repeatDelayScans)) ;
	printf("\n  Key hold scans:         %u", (scaninfo . keyHoldScans)) ;
	printf("\n  Event handler address:  0x%04X:%04X", FP_SEG(scaninfo . codeEntryPoint), FP_OFF(scaninfo . codeEntryPoint)) ;
	printf("\n\n  Membrane keyboard is currently %s.", (membenabledq() ? "enabled" : "disabled")) ;
}

/*---------------------------------------------------------------------------*/

/* Function: setmembraneinfo
   Purpose: Set the membrane keyboard control information.
   Used by: Anyone.
   Returns: Nothing.
   Notes:
   Revision history:
     1) Saturday July 12, 1997 -- 12:20:16.
*/

static void setmembraneinfo(void)

{
	pubmembranekbinfo scaninfo ;


	/* Change a few scan control settings.
	*/
	(scaninfo . repeatRateScans) = 89 ; /* Slow repeat rate for demo. */
	(scaninfo . repeatDelayScans) = 359 ; /* Long old repeat delay for demo. */

	(scaninfo . commandHeader . commandCode) = F_PUBSRVCMEMBSCANINFO ;
	(scaninfo . codeEntryPoint) = newscanhandler ; /* Link in our own membrane keyboard event handler. */
	if(pubservicecall(&scaninfo) < 0)
	{
		fprintf(stderr, "\n\aCan't set membrane keyboard scan control information.\n") ;

		exit(1) ;
	}
}

/*---------------------------------------------------------------------------*/

/* Function: xablemembkb
   Purpose: Xable membrane keyboard scanning.
   Used by: Anyone.
   Returns: Nothing.
   Notes:
   Revision history:
     1) Saturday July 12, 1997 -- 12:20:16.
*/

static void xablemembkb(int enable)

{
	pubmembranekbxable info ;


	(info . commandHeader . commandCode) = F_PUBSRVCMEMBKBXABLE ;
	(info . scanEnable) = !!enable ;
	if(pubservicecall(&info) < 0)
	{
		fprintf(stderr, "\n\aCan't xable membrane keyboard scanning.\n") ;

		exit(1) ;
	}
}

/*---------------------------------------------------------------------------*/

/* Function: membranehooksupportcheck
   Purpose: Make sure that the system ROMBIOS supports the membrane keyboard
	 event/control stuff.
   Used by: Anyone.
   Returns: Nothing.
   Notes:
   Revision history:
     1) Saturday July 12, 1997 -- 12:20:16.
*/

static void membranehooksupportcheck(void)

{
	pubfuncsavailq funcsinfo ;


	(funcsinfo . toggleByte) = 0x55 ;
	(funcsinfo . commandHeader . commandCode) = F_PUBSRVCINFOQ ;
	if(pubservicecall(&funcsinfo) < 0)
	{
		fprintf(stderr, "\n\aPublic services not supported.\n") ;

		exit(1) ;
	}
	if((funcsinfo . toggleByte) != 0xAA)
	{
		fprintf(stderr, "\n\aPublic services code doesn't respond.\n") ;

		exit(1) ;
	}
}

/*---------------------------------------------------------------------------*/

/* Function: main
   Purpose: Program entry point.
   Used by: Program launcher.
   Returns: Success status.
   Notes:
   Revision history:
     1) Saturday July 12, 1997 -- 12:20:16.
*/

int main(unsigned argc, char *argv[])

{
	membranehooksupportcheck() ; /* Make sure that the system ROMBIOS supports the membrane keyboard event/control stuff. */
	displaymembraneinfo() ; /* Display current information. */
	setmembraneinfo() ; /* Set new information. */
	xablemembkb(!0) ; /* Start scanning the membrane keyboard. */

	printf("\nMembrane keyboard enabled.  Going resident...\n") ;
	_dos_keep(0, ((_SS + ((_SP - (_stklen + _heaplen)) / 16)) - _psp)) ;

	return 0 ; /* (Keeps the compiler happy.) */
}

/*---------------------------------------------------------------------------*/
