/*---------------------------------------------------------------------------*/
/*
   "BLTDEMO.C"

   Demonstration program for the Mesa Electronics 4C22 when used with a
     liquid crystal display.  Copies the contents of a file to the LCD
	 display using a bitBLT operation.

   Version 1.1, Monday January 24, 1994 - 22:51:20.
*/
/*---------------------------------------------------------------------------*/
/*
   Compiler: Borland C++, version 3.1.
*/
/*---------------------------------------------------------------------------*/
/*
   Revision history.

   1) Version 1.0, Wednesday December 8, 1993 - 15:40:53.

      Code frozen for version 1.0.


   2) Version 1.1, Monday January 24, 1994 - 22:51:20.

	  Changes from version 1.0:
	  1) Check for level 1 interrupt 0x10 graphics-oriented code support
		 added to function bltimage2disp().

	  2) Demonstration of level 1 interrupt 0x10 code functions
		 F_GRFXSAVEREGNSIZEQ, F_GRFXSAVEREGN, and F_GRFXUNSAVEREGN added to
		 bltimage2disp().
*/
/*---------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <dos.h>
#include <conio.h>
#pragma option -a- /* (Structures must be byte-aligned.) */
#include "sint1a.h"
#include "sint10.h"

/*---------------------------------------------------------------------------*/

/* #defines. */
#define PIXPERLINE 128 /* Number of pixels per image line. */
#define NUMLINES   64  /* Number of image lines. */

/*---------------------------------------------------------------------------*/

/* Local externals.
*/
static unsigned char ImageBuffer[(PIXPERLINE / 8) * NUMLINES] ;

/*---------------------------------------------------------------------------*/

/* Function: sint10ctrlcall
   Purpose: Call special interrupt 0x10 control functions.
   Used by: Anyone.
   Returns: Pass/fail status.
   Notes:
   Revision history:
     1) Wednesday December 8, 1993 - 15:40:53.
*/

signed sint10ctrlcall(void far *parmptr)

{	asm {
				mov		ah,F_SPCLVIDCNTRL
				mov		bx,word ptr parmptr[0]	// Offset of far pointer.
				mov		cx,word ptr parmptr[2]	// Segment of far ptr.
				int		010H					// Invoke the function.
	}
	return (((((int10cmndhdr far *)parmptr) -> errorCode) != E_DISPNONE) ? -1 : 0) ;
}

/*---------------------------------------------------------------------------*/

/* Function: sint10grfxcall
   Purpose: Call special interrupt 0x10 control functions.
   Used by: Anyone.
   Returns: Pass/fail status.
   Notes:
   Revision history:
     1) Wednesday December 8, 1993 - 15:40:53.
*/

signed sint10grfxcall(void far *parmptr)

{	asm {
				mov		ah,F_SPCLVIDGRFX
				mov		bx,word ptr parmptr[0]	// Offset of far pointer.
				mov		cx,word ptr parmptr[2]	// Segment of far ptr.
				int		010H					// Invoke the function.
	}
	return (((((int10cmndhdr far *)parmptr) -> errorCode) != E_GRFXNONE) ? -1 : 0) ;
}

/*---------------------------------------------------------------------------*/

/* Function: sint1actrlcall
   Purpose: Call special interrupt 0x1A control functions.
   Used by: Anyone.
   Returns: Pass/fail status.
   Notes:
   Revision history:
     1) Wednesday December 8, 1993 - 15:40:53.
*/

signed sint1actrlcall(void far *parmptr)

{	asm {
				mov		ah,F_SPCLSYSCNTRL
				mov		bx,word ptr parmptr[0]	// Offset of far pointer.
				mov		cx,word ptr parmptr[2]	// Segment of far ptr.
				int		01AH					// Invoke the function.
	}
	return (((((int1acmndhdr far *)parmptr) -> errorCode) != E_SYSERRNONE) ? -1 : 0) ;
}

/*---------------------------------------------------------------------------*/

/* Function: havespecialint10codeq
   Purpose: Determine whether or not interrupt 0x10 special function code is
     avaible to the programmer.
   Used by: Anyone.
   Returns: Available/not available status.
   Notes:
     -In general, this function need not be called if the CPU is known to
	    support the special interrupt 0x10 functions.
   Revision history:
     1) Wednesday December 8, 1993 - 15:40:53.
*/

int havespecialint10codeq(void)

{	ctrlavailqinfo availinfo ;


	(availinfo . commandHeader . commandCode) = F_DISPCNTRLINFOQ ;
	(availinfo . toggleByte) = 0x55 ;
	if(sint10ctrlcall(&availinfo) < 0)
	{	return !!0 ; /* Error - code not available. */
	}
	else if((availinfo . toggleByte) != 0xAA)
	{	return !!0 ; /* No toggle of toggle byte - code not available. */
	}
	else
	{	return !0 ; /* Code available. */
	}
}

/*---------------------------------------------------------------------------*/

/* Function: havespecialint1acodeq
   Purpose: Determine whether or not interrupt 0x1A special function code is
     avaible to the programmer.
   Used by: Anyone.
   Returns: Available/not available status.
   Notes:
     -In general, this function need not be called if the CPU is known to
	    support the special interrupt 0x1A functions.
   Revision history:
     1) Wednesday December 8, 1993 - 15:40:53.
*/

int havespecialint1acodeq(void)

{	sysfuncsavailq availinfo ;


	(availinfo . commandHeader . commandCode) = F_SYSCNTRLINFOQ ;
	(availinfo . toggleByte) = 0x55 ;
	if(sint1actrlcall(&availinfo) < 0)
	{	return !!0 ; /* Error - code not available. */
	}
	else if((availinfo . toggleByte) != 0xAA)
	{	return !!0 ; /* No toggle of toggle byte - code not available. */
	}
	else
	{	return !0 ; /* Code available. */
	}
}

/*---------------------------------------------------------------------------*/

/* Function: regnsavesizeq
   Purpose: Determine the number of bytes required to save the specified
     rectangular region of the LCD display.
   Used by: Anyone.
   Returns: Number of bytes required, or fail status.
   Notes:
     -No evaluation of the error, if any, occurs; the return condition is
		simply the number of bytes required if >= 0, or failure if < 0.
   Revision history:
     1) Monday January 24, 1994 - 22:51:20.
*/

long regnsavesizeq(rect *theregn)

{	dispregnsavesize regnsize ;


	(regnsize . commandHeader . commandCode) = F_GRFXSAVEREGNSIZEQ ;
	(regnsize . theRegion) = *theregn ;
	return ((sint10grfxcall(&regnsize) >= 0) ? (regnsize . regnSaveBufSize) : -1) ;
}

/*---------------------------------------------------------------------------*/

/* Function: saveregn
   Purpose: Save the specified rectangular region of the LCD display in the
     specified buffer.
   Used by: Anyone.
   Returns: Pass/fail status.
   Notes:
	 -The specified buffer must be large enough to hold the region to be
		saved.
   Revision history:
     1) Monday January 24, 1994 - 22:51:20.
*/

signed saveregn(rect *theregn, unsigned char *savebuf)

{	dispregnsave regninfo ;


	(regninfo . commandHeader . commandCode) = F_GRFXSAVEREGN ;
	(regninfo . theRegion) = *theregn ;
	(regninfo . regnSavePtr) = savebuf ;
	return sint10grfxcall(&regninfo) ;
}

/*---------------------------------------------------------------------------*/

/* Function: restoreregn
   Purpose: Restore the rectangular region of the LCD display from the
     specified buffer.
   Used by: Anyone.
   Returns: Pass/fail status.
   Notes:
	 -The specified buffer must contain a previously-saved region.
   Revision history:
     1) Monday January 24, 1994 - 22:51:20.
*/

signed restoreregn(unsigned char *savebuf)

{	dispregnunsave regnbuf ;


	(regnbuf . commandHeader . commandCode) = F_GRFXUNSAVEREGN ;
	(regnbuf . regnSavePtr) = savebuf ;
	return sint10grfxcall(&regnbuf) ;
}

/*---------------------------------------------------------------------------*/

/* Function: bltimage2disp
   Purpose: Copy the image to the display.
   Used by: Anyone.
   Returns: Nothing.
   Notes:
   Revision history:
     1) Wednesday December 8, 1993 - 15:40:53.
*/

static void bltimage2disp(void)

{	unsigned char *savebufptr ;

	unsigned short gfxlevel ;

	sysvidreroute rerouteinfo ;

	dispavailqinfo availinfo ;

	bitbltinfo bltinfo ;

	crsrxableinfo cursorinfo ;


	/* First, see if we have access to the special interrupt 0x1A functions that
	     we need to reroute video.  (In an environment known to support the
	     special interrupt 0x1A functions, this check need not be performed.)
	*/
	if(!havespecialint1acodeq())
	{	fprintf(stderr, "\n\aSpecial interrupt 0x1A function code is not available.\n") ;

		exit(1) ;
	}

	/* Video rerouting is available; try connecting video output to LCD output. */
	(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ;
	(rerouteinfo . videoDest) = VIDDEST_LCD ; /* Video --> LCD. */
	if(sint1actrlcall(&rerouteinfo) < 0)
	{	fprintf(stderr, "\n\aCouldn't reroute video output to LCD display.\n") ;

		exit(1) ;
	}

	/* Video output functions were successfully rerouted to LCD output functions.
	     We must now determine whether we have access to the functions we want.
	     (In an environment known to support the special interrupt 0x10
	     functions, this check need not be performed.)
	*/
	if(!havespecialint10codeq())
	{	/* Oops!  Our video output functions don't seem to be available.  We must now
		     switch back to standard video output in order to be able to report the
		     error condition to the user.
		*/
		(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ; /* (Redundant.) */
		(rerouteinfo . videoDest) = VIDDEST_VIDEO ; /* Video --> video. */
		sint1actrlcall(&rerouteinfo) ; /* We hope this works, as there's no other way out of this mess. */

		fprintf(stderr, "\n\aSpecial interrupt 0x10 function code is not available.\n") ;

		exit(1) ;
	}

	/* See if the LCD display is active and available. */
	(availinfo . commandHeader . commandCode) = F_GRFXDISPINFOQ ;
	(availinfo . toggleByte) = 0x55 ;
	if(sint10grfxcall(&availinfo) < 0)
	{	/* This should never happen. */
		(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ;
		(rerouteinfo . videoDest) = VIDDEST_VIDEO ; /* Video --> video. */
		sint1actrlcall(&rerouteinfo) ;

		fprintf(stderr, "\n\aCan't determine display availability.\n") ;

		exit(1) ;
	}
	if((availinfo . toggleByte) != 0xAA)
	{	(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ;
		(rerouteinfo . videoDest) = VIDDEST_VIDEO ; /* Video --> video. */
		sint1actrlcall(&rerouteinfo) ;

		fprintf(stderr, "\n\aGraphics code doesn't respond to queries.\n") ;

		exit(1) ;
	}
	if(!(availinfo . isAvailable))
	{	(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ;
		(rerouteinfo . videoDest) = VIDDEST_VIDEO ; /* Video --> video. */
		sint1actrlcall(&rerouteinfo) ;

		fprintf(stderr, "\n\aDisplay isn't available.\n") ;

		exit(1) ;
	}
	gfxlevel = (availinfo . revisionLevel) ; /* Special graphics code revision level. */
	/* The LCD display is active and available. */

	/* Turn off the cursor while the image is being displayed. */
	(cursorinfo  . commandHeader . commandCode) = F_GRFXCRSRXABL ;
	(cursorinfo  . cursorEnable) = !!0 ; /* Turn it off. */
	if(sint10grfxcall(&cursorinfo) < 0) /* (Current cursor state is returned in .cursorEnable.) */
	{	(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ;
		(rerouteinfo . videoDest) = VIDDEST_VIDEO ; /* Video --> video. */
		sint1actrlcall(&rerouteinfo) ;

		fprintf(stderr, "\n\aCan't hide cursor.\n") ;

		exit(1) ;
	}

	/* The top half of the image blinks. */
	(bltinfo . commandHeader . commandCode) = F_GRFXBITBLT ;
	(bltinfo . sourceRect . minCoord . x) = 0 ;
	(bltinfo . sourceRect . minCoord . y) = 0 ;
	(bltinfo . sourceRect . maxCoord . x) = (PIXPERLINE - 1) ;
	(bltinfo . sourceRect . maxCoord . y) = ((NUMLINES / 2) - 1) ;
	(bltinfo . sourcePitch) = (PIXPERLINE / 8) ; /* (Bytes per line.) */
	(bltinfo . sourcePtr) = ImageBuffer ;
	(bltinfo . destPoint . x) = 0 ;
	(bltinfo . destPoint . y) = 0 ;
	(bltinfo . rasterOp) = (RASTEROP_REP | M_RASTEROPBLINK) ; /* Replace/blink. */
	(bltinfo . planeMask) = 0x00 ; /* All planes. (Causes blink if two planes available.) */
	sint10grfxcall(&bltinfo) ; /* (We don't bother to check for errors.) */

	/* Save the lower half of the display for restoration later. */
	savebufptr = 0 ; /* (Used as a flag later.) */
	if(gfxlevel >= 1)
	{	/* This ROMBIOS has interrupt 0x10 level-1 support. */
		long savebuflen ;

		rect r ;


		(r . minCoord . x) = 0 ;
		(r . minCoord . y) = (NUMLINES / 2) ;
		(r . maxCoord . x) = (PIXPERLINE - 1) ;
		(r . maxCoord . y) = (NUMLINES - 1) ;
		if(((savebuflen = regnsavesizeq(&r)) > 0) && (savebuflen < 65520L))
		{	/* We have valid region size information. */
			if((savebufptr = malloc((size_t)savebuflen)) != 0)
			{	/* We have enough memory to save the image. */
				if(saveregn(&r, savebufptr) < 0)
				{	/* Region save failed for some reason. */
					free(savebufptr) ;
					savebufptr = 0 ;
				}
			}
			/* (Else do nothing.) */
		}
		/* (Else do nothing.) */
	}

	/* The lower half of the image blinks. */
	(bltinfo . sourceRect . minCoord . x) = 0 ;
	(bltinfo . sourceRect . minCoord . y) = (NUMLINES / 2) ;
	(bltinfo . sourceRect . maxCoord . x) = (PIXPERLINE - 1) ;
	(bltinfo . sourceRect . maxCoord . y) = (NUMLINES - 1) ;
	(bltinfo . destPoint . x) = 0 ;
	(bltinfo . destPoint . y) = (NUMLINES / 2) ;
	(bltinfo . rasterOp) = RASTEROP_REP ; /* Replace. */
	(bltinfo . planeMask) = 0x02 ; /* Second plane only. (Our demo just assumes that the display has two planes.) */
	sint10grfxcall(&bltinfo) ;

	/* The lower half of the image blinks. */
	(bltinfo . sourceRect . minCoord . x) = 0 ;
	(bltinfo . sourceRect . minCoord . y) = (NUMLINES / 2) ;
	(bltinfo . sourceRect . maxCoord . x) = (PIXPERLINE - 1) ;
	(bltinfo . sourceRect . maxCoord . y) = (NUMLINES - 1) ;
	(bltinfo . sourcePtr) = MK_FP(0xF000, 0xC000) ; /* Some arbitrary address in memory. */
	(bltinfo . destPoint . x) = 0 ;
	(bltinfo . destPoint . y) = (NUMLINES / 2) ;
	(bltinfo . rasterOp) = RASTEROP_REP ; /* Replace. */
	(bltinfo . planeMask) = 0x01 ; /* First plane only. */
	sint10grfxcall(&bltinfo) ;

	/* Yes, Virginia, typing control-C will get out without restoring the cursor.
	*/
	getch() ; /* Wait for a character, but don't display it. */

	/* Put the part of the display we saved back the way it was. */
	if(savebufptr != 0)
	{	restoreregn(savebufptr) ; /* We don't bother to check for errors - the user will notice soon enough. */
		free(savebufptr) ;
	}

	/* Put the cursor back the way it was. */
	if(sint10grfxcall(&cursorinfo) < 0)
	{	(rerouteinfo . commandHeader . commandCode) = F_SYSVIDEOREROUTE ;
		(rerouteinfo . videoDest) = VIDDEST_VIDEO ; /* Video --> video. */
		sint1actrlcall(&rerouteinfo) ;

		fprintf(stderr, "\n\aCan't restore cursor.\n") ;

		exit(1) ;
	}
}

/*---------------------------------------------------------------------------*/

/* Function: main
   Purpose: Program entry point.
   Used by: Program launcher.
   Returns: Never.
   Notes:
   Revision history:
     1) Wednesday December 8, 1993 - 15:40:53.
*/

int main(unsigned argc, char *argv[])

{	size_t bytesread ;

	FILE *fp ;


	if(argc < 2)
	{	fprintf(stderr, "\nPlease specify a file name.") ;
		fprintf(stderr, "\nType <CR> when done viewing.\n") ;

		exit(1) ;
	}
	if((fp = fopen(argv[1], "rb")) == 0)
	{	fprintf(stderr, "\n\aFile not found.\n") ;

		exit(1) ;
	}
	if((bytesread = fread(ImageBuffer, sizeof(unsigned char), sizeof(ImageBuffer), fp)) != sizeof(ImageBuffer))
	{	if(bytesread < sizeof(ImageBuffer))
		{	fprintf(stderr, "\n\aFile < %hu bytes.\n", sizeof(ImageBuffer)) ;
		}
		else
		{	fprintf(stderr, "\n\aFile read error.\n") ;
		}

		exit(1) ;
	}

	bltimage2disp() ;

	exit(0) ;
}

/*---------------------------------------------------------------------------*/
