{ 6117 version of generic flash BIOS write utility for
  Atmel and SST type 5V flash parts (128K) on the FPTX }
program WBIO;
{$R-}
{$I-}

const

BlockSize = 128;                       { the size of Atmel EEPROM blocks }
FileBufferSize = 8192;
BlocksPerBuf = FileBufferSize div BlockSize;
MinBIOS = $8000;
BIOSSize= $20000;

type
FileName  = string[80];
QtBuffer = array[0..32767] of byte;
BufPtr  = ^QtBuffer;

var
TargetSeg : longint;
TargetOff : longint;
BIOSFileName : Filename;
BlocksPerFlash : longint;
BIOSFile : file;
Fbuffer : array [0..FileBufferSize-1] of byte;
Abuffer : array [0..Blocksize-1] of byte;
CheckSum : longint;
BufAPtr : BufPtr;
BufBPtr : BufPtr;
BufCPtr : BufPtr;
BufDPtr : BufPtr;
BlocksInFile : longint;
ReBootAtEnd : boolean;
OldCfg10 : byte;
OldCfg15 : byte;
OldCfg20 : byte;

type
PtrRec = record
           Ofs : word;
           Seg : word;
         end;

procedure DisableInterrupts;
begin
  inline($FA);                                 { CLI }
end;

procedure EnableInterrupts;
begin
  inline($FB);                                 { STI }
end;

{$I beepy}
{$I general}
{$I fpxxlow}
{$I m6117}

procedure Herald;
begin
  writeln;
  writeln('SST BIOS WRITER');
  writeln;
end;

procedure Delay(n: word);
var
iloop,oloop : word;
foobyte : byte;
begin
  for oloop := 0 to n do
  begin
    for iloop := 0 to 250 do foobyte := port[$20];
  end;
end;

procedure Bumout(es:string);
begin
  writeln;
  writeln(chr(7),es);
  halt(2);
end;

procedure ContinueQ(cs:string);
var
response : string;
begin
  writeln;
  write(cs,' Continue y/(n)? ');
  readln(response);
  if upcase(response[1]) <> 'Y' then halt(2);
end;

function OpenForRead(name: Filename): boolean;
begin
  Assign(BIOSFile,Name);
  {$I-}
  Reset(BIOSFile);
  {$I+}
  OpenForRead := IOResult = 0;
end { OpenForRead };

function HexRead(hexstring : string; var hexnumber : longint): boolean;
var
hindex : byte;
charVal : integer;
placeVal : integer;
const
hexChars : string = '0123456789ABCDEF';
begin
  hexNumber :=0;
  HexRead := true;
  for hindex := 1 to length(hexstring) do
  begin
    hexNumber := HexNumber * $10;
    charval := pos(upcase(hexstring[hindex]),hexChars) -1;
    if charval < 0 then
    begin
      HexRead := false;
    end
    else
    begin
      hexnumber := hexnumber + charVal;
    end;
  end;
end;

procedure HexPrint(theNum : longint; nibbles : byte);
var
shiftCount : integer;
const
hexChars : array[0..15] of char = ('0','1','2','3','4','5','6','7',
                                   '8','9','A','B','C','D','E','F');
begin
shiftCount := (nibbles * 4) -4;
  while shiftCount >= 0 do
  begin
    write(hexChars[((thenum shr shiftCount) and $000F)]);
    shiftCount := shiftCount - 4;
  end;
end;

procedure Usage;
begin
  writeln;
  writeln('USAGE: WBIO X000 OOOO FILE [R]');
  writeln('PURPOSE: Writes BIOS file into Atmel/SST Flash BIOS');
  writeln;
  writeln('Where X000 is the (hex) target segment');
  writeln('      OOOO is the (hex) target offset');
  writeln('and   FILE is the file to be written.');
  writeln;
  writeln('      if R option is present,      ');
  writeln('      system is rebooted when done');
  writeln;
  writeln('      Target address MUST be on a 128 byte boundary');
  writeln('      FILE MUST be a multiple of 128 bytes long');
  writeln;
  halt(2);
end;

procedure InvalidHex;
begin
  writeln;
  writeln('Could not extract hex address variable');
  halt(2);
end;

procedure BadTargetAddress;
begin
  writeln;
  writeln('Target address is not in BIOS area');
  halt(2);
end;

function GetParameters : boolean;
var s : string;
begin
  ReBootAtEnd := false;
  if (length(ParamStr(3)) >0)  then
  begin
    if not HexRead(ParamStr(1),TargetSeg) then InvalidHex;
    if TargetSeg < MinBios then BadTargetAddress;
    if not HexRead(ParamStr(2),TargetOff) then InvalidHex;
    BIOSFileName := ParamStr(3);
    if length(ParamStr(4)) > 0 then
    begin
      s:= ParamStr(4);
      if upcase(s[1]) = 'R' then RebootAtEnd := true;
    end;
    GetParameters := true;
  end
  else GetParameters := false;
end;

procedure AllocateBuffers;
begin
  getmem(BufAPtr,Sizeof(QTBuffer));
  getmem(BufBPtr,Sizeof(QTBuffer));
  getmem(BufCPtr,Sizeof(QTBuffer));
  getmem(BufDPtr,Sizeof(QTBuffer));
end;

procedure Setup;
begin
  If not IsFPXX then Bumout('Wrong CPU type!');
  allocateBuffers;
end;

procedure BIOSShadowOff;
begin
  DisableInterrupts;
  port[Page] := BitOff;
  OldCfg15 := ReadConfig($15);
  OldCfg20 := ReadConfig($20);
  WriteConfig($15,$00);
  WriteConfig($20,$84);
end;

procedure BIOSShadowOn;
begin
  WriteConfig($15,OldCfg15);
  WriteConfig($20,OldCfg20);
  port[Page] := BitOn;
  EnableInterrupts;
end;


procedure DumpABuffer;
var
index : byte;

begin
  writeln;
  for index := 0 to BlockSize -1 do
  begin
    HexPrint(ABuffer[index],2);
    write('  ');
    if (index and $0F) = $0F  then writeln;
  end;
end;

procedure FillABuffer(longadd : longint);     { this horrible hack is cause no arrays > 64k bytes }
var
index : byte;
bufsel : byte;
apointer : BufPtr;
longindex : word;
begin
  bufsel := byte(longadd shr 15);              { use bits 15 and 16 to select buffer }
  longindex := word(longadd and $00007FFF);    { only use low 15 bits (0..14) }
  case bufsel of
    0 : apointer := BufAPtr;
    1 : apointer := BufBPtr;
    2 : apointer := BufCPtr;
    3 : apointer := BufDPtr;
  end;
  for index := 0 to BlockSize -1 do
  begin
    ABuffer[index] := apointer^[longindex+index];
  end;
end;

function FillFBuffer : word;
var
numread : word;
begin
  numread := 0;
  blockread(BiosFile,FBuffer,BlocksPerBuf,numread);
  FillFBuffer := numread;
end;

procedure CopyFBufToSBuf(theblk : longint; blks : longint);
var
index : word;
bufsel : byte;
apointer : BufPtr;
longadd : longint;
longindex : word;
csize : word;
begin
  longadd := theblk * BlockSize;
  csize := blks * BlockSize;
  bufsel := byte(longadd shr 15);              { use bits 15 and 16 to select buffer }
  longindex := word(longadd and $00007FFF);    { only use low 15 bits (0..14) }
  case bufsel of
    0 : apointer := BufAPtr;
    1 : apointer := BufBPtr;
    2 : apointer := BufCPtr;
    3 : apointer := BufDPtr;
  end;
  for index := 0 to csize -1 do
  begin
    apointer^[longindex+index] := FBuffer[index];
    CheckSum := (CheckSum + longint(FBuffer[index]));
  end;
end;

function FillSBuffer : longint;
var
block : longint;
numread : longint;
begin
  CheckSum := 0;
  block := 0;
  Reset(BIOSFile,BlockSize);
  repeat
    numread := FillFBuffer;
    if numread > 0 then CopyFBufToSBuf(block,numread);
    block := block + numread;
    if numread = BlocksPerBuf then write(chr(13),'Reading File: ',(block*BlockSize) div 1024,'K  ');
  until numread < BlocksPerBuf;
  FillSBuffer := block;
end;

procedure ReadBIOSFile;
var
fs : longint;
sa : longint;
begin
  if not OpenForRead(BIOSFileName) then
  begin
    Bumout(BIOSFileName+' not found!');
  end
  else
  begin
    Reset(BIOSFile,1);
    fs := filesize(BIOSFile);
    if fs > BIOSSize then Bumout('128K or < only');
    if fs and $0000007F <> 0 then Bumout('Wierd size ... abend');
    sa := ((TargetSeg * 16) + TargetOff);
    if (sa + fs) > $100000 then Bumout('Over the top!');
  end;
  BlocksInFile := FillSBuffer;
  writeln;
  write('Start address is ');
  HexPrint(sa,5);
  writeln('H');
  write('File size is ');
  HexPrint(BlocksInFile * BlockSize,5);
  writeln('H bytes');
  write('File checksum is ');
  HexPrint(CheckSum,8);
  writeln('H');
end;

procedure WriteAtmelSector(source : longint; target: longint);
var
rdata : byte;
ldata : byte;
chipseg,tword,tofs,tseg : word;
loopcount : longint;
index : byte;
writeok : boolean;
chipadd : longint;
begin
  FillABuffer(source);
  writeok := false;
  tseg := word(target shr 4);
  tofs := 0;
  chipseg := tseg and $F000;
  mem[chipseg:$5555] :=$AA;
  mem[chipseg:$2AAA] :=$55;
  mem[chipseg:$5555] :=$A0;
  for index := 0 to BlockSize-1 do
  begin
    mem[tseg:tofs+index] := ABuffer[index];
  end;
  ldata := ABuffer[BlockSize-1];
  rdata := ldata xor $FF;
  loopcount := 0;
  while ((rdata <> ldata) and (loopcount < 10000)) do
  begin
    rdata := mem[tseg:tofs+BlockSize-1];
    loopcount := loopcount +1;
  end;
  if loopcount < 9999 then writeok := true else
  begin
    BIOSShadowOn;
    write('Error at source address ');
    HexPrint(source,8);
    write(' and target address ');
    HexPrint(target,8);
   { DumpABuffer; }
    Bumout('Don''t REBOOT, retry!');
  end;
end;

procedure DoTransfer;
var
block : longint;
target : longint;
source : longint;
begin
  ContinueQ('Re-Write BIOS ...');
  target := TargetSeg shl 4 + TargetOff;
  source := 0;
  BiosShadowOff;
  for block := 0 to BlocksInFile -1 do
  begin
    WriteAtmelSector(source,target);
    WarbleDog;
    source := source + BlockSize;
    target := target + BlockSize;
    if (block and 31) = 0 then DoBeep(400,10);
  end;
  BiosShadowOn;
end;

begin { main }
  if not GetParameters then usage else
  begin
    Herald;
    Setup;
    ReadBIOSFile;
    DoTransfer;
    if RebootAtEnd then Reboot else writeln ('Done! - Reboot now!');
  end;
end.
