{ 2d mono Widgets using MESA BIOS LCD graphic functions 1.5 rev or > BIOS only }

type

PictureRec = record
  XDim : integer;
  YDim : integer;
  PicPtr : pointer;
end;

const

{ Include our function numbers }

{$I SInt1A}

{$I SInt10}

{ Include some simple bitmaps }

{$I ptrleft}
{$I ptrright}
{$I ptrdown}
{$I ptrup}
{$I unhappy}

ValidSeal =   $CAFEBABE;
InvalidSeal = $DEADFACE;

type

Edge = (Top,Bottom,Right,Left);

TextPos = (LeftJustify,CenterJustify,RightJustify,TopJustify,BottomJustify);

PointRec = record
  X : integer;
  Y : integer;
end;

RectRec = record
  UL : PointRec;
  LR : PointRec;
end;

GPtrRec = record
  Ofs : word;
  Smt : word;
end;

FuncHdrRec = record
  CommandCode : byte;
  ErrorCode : byte;
end;

GetSetByteRec =
record
  FuncHdr : FuncHdrRec;
  Data : byte;
end;

GetSetWordRec =
record
  FuncHdr : FuncHdrRec;
  Data : word;
end;

DisplayDimRec =
record
  FuncHdr : FuncHdrRec;
  DispDims : PointRec;
  DispPlanes : byte;
end;

EraseRegionRec =
record
  FuncHdr : FuncHdrRec;
  Region : RectRec;
end;

SetFillPatRec =
record
  FuncHdr : FuncHdrRec;
  PatPoint : Pointer;
end;

SaveRegionRec =
record
  FuncHdr : FuncHdrRec;
  Region : RectRec;
  MemPtr : Pointer;
end;

RestRegionRec =
record
  FuncHdr : FuncHdrRec;
  MemPtr : Pointer;
end;

GetRegionBuffSizeRec =
record
  FuncHdr : FuncHdrRec;
  Region : RectRec;
  BuffSize : longint;
end;

SetTTYWinRec =
record
  FuncHdr : FuncHdrRec;
  TTYWindow : RectRec;
end;

DrawCharRec =
record
  FuncHdr : FuncHdrRec;
  DestPoint : PointRec;
  TheChar : word;
  RasterOp : byte;
  PlaneMask : byte;
end;

KeyBoardRouteRec  =
record
  FuncHdr : FuncHdrRec;
  KBSource : byte;
  MembraneToo : byte;
end;

BaudRateSelectRec =
record
  FuncHdr : FuncHdrRec;
  RateSelector : byte;
end;

MesaGrContRec =
record
  FuncHdr : FuncHdrRec;
  ToggleByte : byte;
  RevisionLevel : word;
end;

MesaSysContRec =
record
  FuncHdr : FuncHdrRec;
  ToggleByte : byte;
  RevisionLevel : word;
end;

MesaGrAvailRec =
record
  FuncHdr : FuncHdrRec;
  ToggleByte : byte;
  IsAvail    : byte;
  RevisionLevel : word;
end;

DrawLineRec =
record
  FuncHdr : FuncHdrRec;
  StartPoint : PointRec;
  EndPoint : PointRec;
  LineColor : byte;
  SkipFirst : byte;
  SkipLast : byte;
  RasterOp : byte;
  PlaneMask : byte;
end;

FillRegionRec =
record
  FuncHdr : FuncHdrRec;
  Region : RectRec;
  PatPoint : pointer;
  RasterOp : byte;
  PlaneMask : byte;
end;

GetSetDispModeRec =
record
  FuncHdr : FuncHdrRec;
  InvertDisp : byte;
end;

FontInfoRec =
record
  CharWidth : byte;
  CharHeight : byte;
  BytesPerLine : byte;
  BytesPerChar : byte;
  CharsInFont : word;
  FontPtr : pointer;
end;

GetSetFontInfoRec =
record
  FuncHdr : FuncHdrRec;
  FontInfo : FontInfoRec;
end;

TTYStateRec =
record
  TTYUpperLeft : PointRec;
  TTYDim : PointRec;
  FontInfo : FontInfoRec;
  CursorDimX : byte;
  CursorDimY : byte;
  CursorOffsetX : byte;
  CursorOffsetY : byte;
  CursorLoc : PointRec;
  EraseQ : byte;
end;

GetSetTTYStateRec =
record
  FuncHdr : FuncHdrRec;
  TTYState : TTYStateRec;
end;

AtoDTypeRec =
record
  FuncHdr : FuncHdrRec;
  StringPtr : pointer;
  NumChans : word;
end;

ReadCardTempRec =
record
  FuncHdr : FuncHdrRec;
  CardTemp : word;
end;

ReadAtoDRec =
record
  FuncHdr : FuncHdrRec;
  isTimeCritical : byte; {0 if not ...}
  ChannelNumber : byte;
  AtoDData : word;
end;

GetSetFontInfoByNumRec =
record
  FuncHdr    : FuncHdrRec;
  FontNumber : byte;
  FontInfo   : FontInfoRec;
end;

DisplayInfoRec =
record
  DisplayDim : PointRec;
  NybblesPerLine : word;
  Cl1Start : word;
  Cl1End : word;
  FLMStartOffset : word;
  FLMEndOffset : word;
  NumPlanes : byte;
  BaseParagraph : word;
  InvertDisp : byte;
end;

GetSetDisplayInfoRec =
record
  FuncHdr : FuncHdrRec;
  DisplayInfo : DisplayInfoRec;
end;

GetSetKBScanInfoRec =
record
  FuncHdr : FuncHdrRec;
  ClickEnable : byte;      { Enable generation of key clicks by default (ROMBIOS-resident) scan event handler. }
  ClickTicks : word;       { Duration of system-generated key click beeps. }
  ClickDivisor : word;     { Key click beep divisor for system-generated clicks. }
  DebounceTicks : word;    { Debounce time. }
  RepeatRateTicks : word;  { Repeat event generation rate. }
  RepeatDelayTicks : word; { Initial repeat event generation delay. 0 for no repeat }
  CodeEntryPoint : pointer;{ Address of membrane event handler. }
  NumKeys : word;          { Number of keys in scan matrix. This field is read-only. }
  KeyHoldTime : byte;      { Key up recognition delay time. (Noise filter.) }
  Reserved : array [1..3] of byte;      { Currently unused/reserved. }
end;

BltRec =
record
  FuncHdr : FuncHdrRec;
  SourceRect : RectRec;
  SourcePitch : word;
  SourcePtr : pointer;
  DestPoint : PointRec;
  DestPitch : word;
  DestPtr : pointer;
  RasterOp : byte;
  PlaneMask : byte;
end;

UnBltRec =
record
  FuncHdr : FuncHdrRec;
  DestRect : RectRec;
  DestPitch : word;
  DestPtr : pointer;
  SourcePoint : PointRec;
  RasterOp : byte;
  PlaneMask : byte;
end;

MiscRec =
record
  FuncHdr : FuncHdrRec;
  Parms : array[1..16] of byte;
end;

ScaleInfoRec =
record
  Scale         : real;
  Offset        : integer;
  MinReading    : integer;
  MaxReading    : integer;
  TicInterval   : integer;
  MeterType     : (Vertical,Horizontal);
  ScaleRect     : RectRec;
end;

{**************************************************************************}

BarGraphPtr = ^BarGraphRec;

BarGraphRec =
record
  Seal      : longint;
  Outline   : RectRec;
  CurBar    : RectRec;
  DeltaBar  : RectRec;
  BarLength : word;
  ScaleInfo : ScaleInfoRec;
  BufPtr    : Pointer;
  BufSize   : word;
end;

EdgeMeterPtr = ^EdgeMeterRec;

EdgeMeterRec =
record
  Seal          : longint;
  Outline       : RectRec;
  CurPtr        : word;
  PtrRect       : RectRec;
  PtrSize       : word;
  PtrRangeRect  : RectRec;
  BltOffset     : word;
  PrevData      : integer;
  ScaleInfo     : ScaleInfoRec;
  BufPtr        : Pointer;
  BufSize       : word;
end;

TextWindowPtr = ^TextWindowRec;

TextWindowRec =
record
  Seal              : longint;
  Outline           : RectRec;
  SaveUnderBufSize  : word;
  SaveUnderBufPtr   : pointer;
  SaveUnderValid    : boolean;
  SaveSelfBufSize   : word;
  SaveSelfBufPtr    : pointer;
  SaveSelfValid     : boolean;
  TTYStateInfo      : TTYStateRec;
  HasFocus          : boolean;
  WeStoleFocus      : boolean;
  StolenFrom        : TextWindowPtr;
  Blinkf            : boolean;
end;

TitledBoxPtr = ^TitledBoxRec;

TitledBoxRec =
record
  Seal          : longint;
  Outline       : RectRec;
  TitleBox      : RectRec;
  DataBox       : RectRec;
  TitleText     : string;
  TitleLocation : Edge;
  TextPosition  : TextPos;
  BufPtr        : Pointer;
  BufSize       : word;
end;

GStateRec =
record
  DisplayInfo     : DisplayInfoRec;
  ScreenRect      : RectRec;
  CurTTYState     : TTYStateRec;
  OldTTYState     : TTYStateRec;
  TopWindowPtr    : TextWindowPtr;
  GraphicFontInfo : FontInfoRec;
  Font0Info       : FontInfoRec;
  Font1Info       : FontInfoRec;
  FGColor         : byte;
  BGColor         : byte;
  FGPat           : pointer;
  BGPat           : pointer;
  PlaneMask       : byte;
  FGRasterOp      : byte;
  BGRasterOp      : byte;
  OldVideoDest    : byte;
  OldKeySource    : byte;
  OldMembToo      : byte;
end;

var GState : GStateRec;

const

SolidBlackPat       : array[0..7] of byte = ($FF,$FF,$FF,$FF,$FF,$FF,$FF,$FF);
SolidWhitePat       : array[0..7] of byte = ($00,$00,$00,$00,$00,$00,$00,$00);
Halftone50Pat       : array[0..7] of byte = ($55,$AA,$55,$AA,$55,$AA,$55,$AA);
Halftone50BigPat    : array[0..7] of byte = ($33,$33,$CC,$CC,$33,$33,$CC,$CC);
Halftone50BiggerPat : array[0..7] of byte = ($0F,$0F,$0F,$0F,$F0,$F0,$F0,$F0);
Halftone25Pat       : array[0..7] of byte = ($22,$88,$22,$88,$22,$88,$22,$88);
XzuzPat             : array[0..7] of byte = ($81,$42,$24,$18,$18,$24,$42,$81);
CrossPat            : array[0..7] of byte = ($FF,$88,$88,$88,$FF,$88,$88,$88);

{ ***** low level point and rectangle utility routines ***** }

procedure MakeRect(ulx,uly,lrx,lry : integer; var therect : RectRec);
{ Make a rectangle from corners }
begin
  therect.UL.X := ulx;
  therect.UL.Y := uly;
  therect.LR.X := lrx;
  therect.LR.Y := lry;
end;

procedure CandDFromRect(rect : RectRec; var corner,dim : PointRec);
{ Make corner and dimensions from rectangle }
begin
  corner.X := rect.UL.X;
  corner.Y := rect.UL.Y;
  dim.X    := rect.LR.X - rect.UL.X +1;
  dim.Y    := rect.LR.Y - rect.UL.Y +1;
end;

procedure RectFromCandD(var rect : RectRec; corner,dim : PointRec);
{ Make rectangle from corner and dimensions }
begin
  rect.UL.X :=  corner.X;
  rect.UL.Y :=  corner.Y;
  rect.LR.X :=  corner.X + dim.X -1;
  rect.LR.Y :=  corner.Y + dim.Y -1;
end;

procedure MakePoint(x,y : integer; var thepoint : PointRec);
{ Make a point from x,y }
begin
  thepoint.X := x;
  thepoint.Y := y;
end;

procedure InsetRect(srect : RectRec; var drect : RectRec;delx,dely : integer);
{ grow or shrink a rectangle symmetrically by delx and dely pixels. - del means smaller }
begin
  drect.UL.X := srect.UL.X - delx;
  drect.UL.Y := srect.UL.Y - dely;
  drect.LR.X := srect.LR.X + delx;
  drect.LR.Y := srect.LR.Y + dely;
end;

procedure InsetEdge(srect : RectRec; var drect : RectRec; theedge : edge ;del : integer);
{ Grow or shrink a rectangle edge by 'del' pixels + del means bigger - del means smaller }
begin
  drect := srect;
  case theedge of
    Top :
    begin
      drect.UL.Y := srect.UL.Y - del;
    end;
    Bottom :
    begin
      drect.LR.Y := srect.LR.Y + del;
    end;
    Left :
    begin
       drect.UL.X := srect.UL.X - del;
    end;
    Right :
    begin
       drect.LR.X := srect.LR.X + del;
    end;
  end;
end;

procedure MoveRect(var rect : RectRec;x,y: integer);
{ Move a rectangle by x,y }
begin
  rect.UL.X := rect.UL.X + x;
  rect.UL.Y := rect.UL.Y + y;
  rect.LR.X := rect.LR.X + x;
  rect.LR.Y := rect.LR.Y + y;
end;

function RectXSize(rect : RectRec) : word;
{ Get x dimension of rectangle -- a single pixel is a unit rectangle!}
begin
  RectXSize := rect.LR.X - rect.UL.X + 1;
end;

function RectYSize(rect : RectRec) : word;
{ Get y dimension of rectangle -- a single pixel is a unit rectangle!}
begin
  RectYSize := rect.LR.Y - rect.UL.Y + 1;
end;

procedure SplitRectX(srect :RectRec; var drect1,drect2 : RectRec; split : word);
{ Split srect into 2 new rectangles. Dividing line is determined by split }
{ split must be greater than 0 and less than the width of srect           }
begin
  drect1.UL := srect.UL;
  drect1.LR.X := srect.UL.X + split;
  drect1.LR.Y := srect.LR.Y;
  drect2.UL.X := srect.UL.X + split;
  drect2.UL.Y := srect.UL.Y;
  drect2.LR := srect.LR;
end;

procedure SplitRectY(srect :RectRec; var drect1,drect2 : RectRec; split : word);
{ Split srect into 2 new rectangles. Dividing line is determined by split }
{ split must be greater than 0 and less than the height of srect          }
begin
  drect1.UL := srect.UL;
  drect1.LR.X := srect.LR.X;
  drect1.LR.Y := srect.UL.Y + split;
  drect2.UL.X := srect.UL.X;
  drect2.UL.Y := srect.UL.Y + split;
  drect2.LR := srect.LR;
end;

procedure CenterOfRect(srect :RectRec;var p :PointRec);
{ Return the center of a rectangle }
begin
  p.X :=(srect.UL.X + srect.LR.X) div 2;
  p.Y :=(srect.UL.Y + srect.LR.Y) div 2;
end;

function CenterOfRectX(srect :RectRec) : integer;
{ Return the center xdim of a rectangle }
begin
  CenterOfRectX :=(srect.UL.X + srect.LR.X) div 2;
end;

function CenterOfRectY(srect :RectRec) : integer;
{ Return the center Ydim of a rectangle }
begin
  CenterOfRectY :=(srect.UL.Y + srect.LR.Y) div 2;
end;

procedure DivideRectX(srect :RectRec; var drect : RectRec; pieces,startp,endp : word);
{ Return a rectangle sliced from a source rectangle. Pieces determines width }
{ of slices as fraction of the source rectangles width. Startp and endp      }
{ determine size and position of slice returned in units of pieces.          }
{ For example : DivideRectX(oldrect,newrect,10,1,5) would return a rectangle }
{ that was the left hand half of oldrect (pieces 1 through 5 of 10 total)    }
{ DivideRectX(oldrect,newrect,3,2,2) would return a rectangle that was the   }
{ middle third of oldrect (piece 2 of pieces 1,2,3)                          }
var
piecesize : word;
begin
  piecesize := RectXSize(srect) div pieces;
  drect.UL.X := srect.UL.X  + (startp -1) * piecesize;
  drect.LR.X := srect.UL.X  + (endp     ) * piecesize;
  drect.UL.Y := srect.UL.Y;
  drect.LR.Y := srect.LR.Y;
end;

procedure DivideRectY(srect :RectRec; var drect : RectRec; pieces,startpiece,endpiece : word);
{ Return a rectangle sliced from a source rectangle. Pieces determines height}
{ of slices as fraction of the source rectangles height. Startp and endp      }
{ determine size and position of slice returned in units of pieces.          }
{ For example : DivideRectY(oldrect,newrect,10,1,5) would return a rectangle }
{ that was the top half of oldrect (pieces 1 through 5 of 10 total)          }
{ DivideRectX(oldrect,newrect,3,2,2) would return a rectangle that was the   }
{ middle third of oldrect (piece 2 of pieces 1,2,3)                          }
var
piecesize : word;
begin
  piecesize := RectYSize(srect) div pieces;
  drect.UL.Y := srect.UL.Y  + (startpiece -1) * piecesize;
  drect.LR.Y := srect.UL.Y  + (endpiece     ) * piecesize;
  drect.UL.X := srect.UL.X;
  drect.LR.X := srect.LR.X;
end;

procedure PrintRect(srect : RectRec);
{ Debug utility to print rectangle points }
begin
  write(srect.UL.X,',',srect.UL.Y,',',srect.LR.X,',',srect.LR.Y);
end;

{ ***** These are the low level functions that call the graphic or ***** }
{ ***** control routines in the MESA 4C22,FPWX or FPCX BIOS        ***** }

procedure MesaGrFunc(p: pointer);
begin
  asm
    mov ah,F_SPCLVIDGRFX
    mov cx,GPtrRec[p].Smt
    mov bx,GPtrRec[p].Ofs
    int 10h
  end;
end;

procedure MesaGrContFunc(p: pointer);
begin
  asm
    mov ah,F_SPCLVIDCNTRL
    mov cx,GPtrRec[p].Smt
    mov bx,GPtrRec[p].Ofs
    int 10h
  end;
end;

procedure MesaSysContFunc(p: pointer);
begin
  asm
    mov ah,F_SPCLSYSCNTRL;
    mov cx,GPtrRec[p].Smt
    mov bx,GPtrRec[p].Ofs
    int 1Ah
  end;
end;

procedure MesaSysSetupFunc(p: pointer);
begin
  asm
    mov ah,F_SYSSETUP;
    mov cx,GPtrRec[p].Smt
    mov bx,GPtrRec[p].Ofs
    int 1Ah
  end;
end;

function ReadAtoD(channel : byte) : word;
var
atodrec : ReadAtoDRec;
begin
  with atodrec do
  begin
    FuncHdr.CommandCode := F_SYSATODRAWREAD;
    ChannelNumber := channel;
    IsTimeCritical := 0; { no, wait if busy}
    { Should never be busy with normal (synchronous ) operation }
    { but could be busy if called from an interrupt routine     }
    { not recommended )-: }
    MesaSysContFunc(@atodrec);
    ReadAtoD := AtoDData;
  end;
end;

function ReadCardTemp : word;
var
rtemprec : ReadCardTempRec;
begin
  with rtemprec do
  begin
    FuncHdr.CommandCode := F_SYSTEMPSENSE;
    MesaSysContFunc(@rtemprec);
    ReadCardTemp := CardTemp;
  end;
end;

function QKeyPressed : boolean;
var flag : boolean;
begin
  asm
    mov ah,01
    int $16
    mov flag,false
    jz @nokey
    mov flag,true
    @nokey:
  end;
  QKeyPressed := flag;
end;

function QKeyRead : word;
var temp : word;
begin
  asm
    mov ah,00;
    int $16
    mov temp,ax;
  end;
  QKeyRead := temp;
end;

function PushOnKBFIFO(achar : byte) : boolean;
var rc : byte;
begin
  asm
    mov ah,05
    mov ch,$1
    mov cl,achar
    int($16);
    mov rc,al
  end;
  if rc = 1 then PushOnKBFIFO := false else PushOnKBFIFO := true ;
end;

procedure SCUBaudRateSelect(selector : byte);
var brec : BaudRateSelectRec;
begin
  brec.FuncHdr.CommandCode := F_SYSSCUBAUDSEL;
  brec.RateSelector := selector;
  MesaSysContFunc(@brec);
end;

function GetSCUBaudRateSelect : byte;
var brec : BaudRateSelectRec;
begin
  brec.FuncHdr.CommandCode := F_SYSSCUBAUDSELQ;
  MesaSysContFunc(@brec);
  GetSCUBaudRateSelect := brec.RateSelector;
end;

function GetContrast : word;
var GetSetContrastInfo : GetSetWordRec;
begin
  GetSetContrastInfo.FuncHdr.CommandCode := F_DISPCONTRASTGET;
  MesaGrContFunc(@GetSetContrastInfo);
  GetContrast := GetSetContrastInfo.Data;
end;

procedure SetContrast(contrast : word);
var GetSetContrastInfo : GetSetWordRec;
begin
  GetSetContrastInfo.FuncHdr.CommandCode := F_DISPCONTRASTSET;
  GetSetContrastInfo.Data := contrast;
  MesaGrContFunc(@GetSetContrastInfo);
end;

function GetBrightness : word;
var GetSetBrightnessInfo : GetSetWordRec;
begin
  GetSetBrightnessInfo.FuncHdr.CommandCode := F_DISPBKLTTENSITYGET;
  MesaGrContFunc(@GetSetBrightnessInfo);
  GetBrightness := GetSetBrightnessInfo.Data;
end;

procedure SetBrightness(Brightness : word);
var GetSetBrightnessInfo : GetSetWordRec;
begin
  GetSetBrightnessInfo.FuncHdr.CommandCode := F_DISPBKLTTENSITYSET;
  GetSetBrightnessInfo.Data := Brightness;
  MesaGrContFunc(@GetSetBrightnessInfo);
end;

function GetDispTimeout : byte;
var GetSetDispTimeoutInfo : GetSetByteRec;
begin
  GetSetDispTimeoutInfo.FuncHdr.CommandCode := F_DISPTIMEOUTGET;
  MesaGrContFunc(@GetSetDispTimeoutInfo);
  GetDispTimeout := GetSetDispTimeoutInfo.Data;
end;

procedure SetDispTimeout(DispTimeout : byte);
var GetSetDispTimeoutInfo : GetSetbyteRec;
begin
  GetSetDispTimeoutInfo.FuncHdr.CommandCode := F_DISPTIMEOUTSET;
  GetSetDispTimeoutInfo.Data := DispTimeout;
  MesaGrContFunc(@GetSetDispTimeoutInfo);
end;

procedure SaveSetup;
var ControlInfo : FuncHdrRec;
begin
  ControlInfo.CommandCode := F_SYSSETUPSAVE;
  MesaSysSetupFunc(@ControlInfo);
end;

procedure SetVidDest(dest : byte);
var vidd : GetSetByteRec;
begin
  vidd.FuncHdr.CommandCode := F_SYSVIDEOREROUTE;
  vidd.Data := dest;
  MesaSysContFunc(@vidd);
end;

function GetVidDest : byte;
var vidd : GetSetByteRec;
begin
  vidd.FuncHdr.CommandCode := F_SYSVIDEOSOURCEQ;
  MesaSysContFunc(@vidd);
  GetVidDest :=  vidd.Data;
end;

procedure LCDOut;
begin
  SetVidDest(VIDDEST_LCD);
end;

procedure StubLCDOut;
begin
  SetVidDest(VIDDEST_STUB);
end;

procedure SerialOut;
begin
  SetVidDest(VIDDEST_SERIAL);
end;

procedure VideoOut;
begin
  SetVidDest(VIDDEST_VIDEO);
end;

procedure SetDispInvert(invert : boolean);
var dmrec : GetSetDispModeRec;
begin
  with dmrec do
  begin
    FuncHdr.CommandCode := F_DISPMODESET;
    if invert = true then InvertDisp := $FF else InvertDisp := $00;
  end;
  MesaGrFunc(@dmrec);
end;

function DispInvertQ : boolean;
var dmrec : GetSetDispModeRec;
begin
  with dmrec do
  begin
    FuncHdr.CommandCode := F_DISPMODEGET;
    MesaGrFunc(@dmrec);
    if InvertDisp <> 0 then DispInvertQ := true else DispInvertQ := false;
  end;
end;

procedure SleepCursor;
var amiscrec : MiscRec;
begin
  { Note that it may be desirable ( or even necessary) to disable (sleep)  }
  { the cursor when doing high speed or time critical operations. This is  }
  { because the LCD cursor blink is an interrupt driven background task    }
  { that may delay time critical code. Another possibility is to mask the  }
  { frame interrupt (normally IRQ5) at the 8259                            }
  amiscrec.FuncHdr.CommandCode :=F_GRFXCRSRXABL;
  amiscrec.Parms[1] := $00;
  MesaGrFunc(@amiscrec);
end;

procedure ClearScreen;
var amiscrec : MiscRec;
begin
  { Note that this clears only the TTY region }
  { and that the fill pattern must be set previously! }
  amiscrec.FuncHdr.CommandCode :=F_GRFXERASETTY;
  MesaGrFunc(@amiscrec);
end;

procedure WakeCursor;
var amiscrec : MiscRec;
begin
  amiscrec.FuncHdr.CommandCode :=F_GRFXCRSRXABL;
  amiscrec.Parms[1] := $FF;
  MesaGrFunc(@amiscrec);
end;

procedure GetKeySource(var ks,mt : byte);
var kbrec : KeyBoardRouteRec;
begin
  kbrec.FuncHdr.CommandCode := F_SYSKBSOURCEQ;
  MesaSysContFunc(@kbrec);
  ks := kbrec.KBSource;
  mt := kbrec.MembraneToo;
end;

procedure SetKeySource(ks,mt : byte);
var kbrec : KeyBoardRouteRec;
begin
  kbrec.FuncHdr.CommandCode := F_SYSKBREROUTE;
  kbrec.KBSource := ks;
  kbrec.MembraneToo := mt;
  MesaSysContFunc(@kbrec);
end;

procedure KeyPadOnly;
var kbrec : KeyBoardRouteRec;
begin
  kbrec.FuncHdr.CommandCode := F_SYSKBREROUTE;
  kbrec.KBSource := KBSRC_OFF;
  kbrec.MembraneToo := $FF;
  MesaSysContFunc(@kbrec);
end;

procedure SerialInOnly;
var kbrec : KeyBoardRouteRec;
begin
  kbrec.FuncHdr.CommandCode := F_SYSKBREROUTE;
  kbrec.KBSource := KBSRC_SERIAL;
  kbrec.MembraneToo := $00;
  MesaSysContFunc(@kbrec);
end;

procedure XTKBOnly;
var kbrec : KeyBoardRouteRec;
begin
  kbrec.FuncHdr.CommandCode := F_SYSKBREROUTE;
  kbrec.KBSource := KBSRC_KEYBOARD;
  kbrec.MembraneToo := $00;
  MesaSysContFunc(@kbrec);
end;

{ both is no go for now on FPWX and 4C22 }
procedure BothKBs;
var kbrec : KeyBoardRouteRec;
begin
  kbrec.FuncHdr.CommandCode := F_SYSKBREROUTE;
  kbrec.KBSource := KBSRC_KEYBOARD;
  kbrec.MembraneToo := $FF;
  MesaSysContFunc(@kbrec);
end;

function MesaGrFuncsAvailq : boolean;
var grfuncs : MesaGrAvailRec;
begin
  { Good idea to make sure graphic functions are available }
  { before diving in...}
  MesaGrFuncsAvailq := false;
  grfuncs.FuncHdr.CommandCode := F_GRFXDISPINFOQ;
  grfuncs.ToggleByte := $55;
  MesaGrFunc(@grfuncs);
  if grfuncs.ToggleByte = $AA then
  MesaGrFuncsAvailq := true;
end;

function MesaGrFuncsRev : word;
var
grfuncs : MesaGrAvailRec;
begin
  { Check the BIOS graphic code rev level }
  grfuncs.RevisionLevel := 0;
  grfuncs.FuncHdr.CommandCode := F_GRFXDISPINFOQ;
  MesaGrFunc(@grfuncs);
  MesaGrFuncsRev := grfuncs.RevisionLevel;
end;

function GetRegionBuffSize(theregion : RectRec): longint;
var bufsizerec : GetRegionBuffSizeRec;
begin
  with bufsizerec do
  begin
    FuncHdr.CommandCode := F_GRFXSAVEREGNSIZEQ;
    Region := theregion;
    MesaGrFunc(@bufsizerec);
    GetRegionBuffSize := BuffSize;
  end;
end;

procedure SaveRegion(theregion : RectRec;p: pointer);
var svrrec : SaveRegionRec;
begin
  with svrrec do
  begin
    FuncHdr.CommandCode :=F_GRFXSAVEREGN;
    Region := theregion;
    MemPtr := p;
  end;
  MesaGrFunc(@svrrec);
end;

procedure RestoreRegion(p: pointer);
var resrrec : RestRegionRec;
begin
  with resrrec do
  begin
    FuncHdr.CommandCode := F_GRFXUNSAVEREGN;
    MemPtr := p;
  end;
  MesaGrFunc(@resrrec);
end;

procedure GetDispSize(var xdim,ydim : word;var nplanes : byte);
{ return the display size as x and y dimensions and number of planes }
var dimrec : DisplayDimRec;
begin
  with dimrec do
  begin
    FuncHdr.CommandCode :=F_GRFXDISPDIMQ;
    MesaGrFunc(@dimrec);
    xdim := DispDims.X;
    ydim := DispDims.Y;
    nplanes := DispPlanes;
  end;
end;

procedure GetDispRect(var drect :RectRec);
{ return the full display size as a rectangle }
var
xsize,ysize : word;
planes: byte;
begin
  GetDispSize(xsize,ysize,planes);
  drect.UL.X := 0;
  drect.UL.Y := 0;
  drect.LR.X := xsize -1;
  drect.LR.Y := ysize -1;
end;

procedure EraseRegion(therect : RectRec);
{ Erase a rectangular region to the current pattern }
var errec : EraseRegionRec;
begin
  with errec do
  begin
    FuncHdr.CommandCode :=F_GRFXERASEREGN;
    Region := therect;
    MesaGrFunc(@errec);
  end;
end;

procedure FillRegion(therect : RectRec;pp : pointer;ro,pm :byte);
{ fill a rectangular region with the specified pattern, rasterop and planemask }
var frrec : FillRegionRec;
begin
  with frrec do
  begin
    FuncHdr.CommandCode :=F_GRFXFILLREGN;
    Region := therect;
    PatPoint := pp;
    RasterOp := ro;
    PlaneMask := pm;
    MesaGrFunc(@frrec);
  end;
end;

procedure SetFillPattern(p:pointer);
{ set the global fill (erase) pattern }
var fprec : SetFillPatRec;
begin
  with fprec do
  begin
    FuncHdr.CommandCode :=F_GRFXPATSET;
    PatPoint := p;
  end;
  MesaGrFunc(@fprec);
end;

procedure GetTTYState(var ttstate : TTYStateRec);
var
gsttystate : GetSetTTYStateRec;
begin
  with gsttystate do
  begin
    FuncHdr.CommandCode :=F_GRFXTTYSTATEGET;
  end;
  MesaGrFunc(@gsttystate);
  ttstate := gsttystate.TTYState;
end;

function GraphicsRevOK : boolean;
var
gsttystate : GetSetTTYStateRec;
begin
  GraphicsRevOK := false;
  with gsttystate do
  begin
    FuncHdr.CommandCode :=F_GRFXTTYSTATEGET;
    MesaGrFunc(@gsttystate);
    if FuncHdr.ErrorCode = E_DISPNONE then GraphicsRevOK := true;
  end;
end;

procedure SetTTYState(ttstate : TTYStateRec; clearit : boolean);
var
gsttystate : GetSetTTYStateRec;
begin
  if clearit then ttstate.EraseQ := $FF else ttstate.EraseQ := $00;
  gsttystate.TTYState := ttstate;
  with gsttystate do
  begin
    GState.CurTTYState := gsttystate.TTYState;  { shadow }
    FuncHdr.CommandCode :=F_GRFXTTYSTATESET;
  end;
  MesaGrFunc(@gsttystate);
end;

procedure GetBIOSFontInfo(var thefontinfo : FontInfoRec;num : byte);
{ get the various BIOS font parameters }
var gsfontinfobn :GetSetFontInfoByNumRec;
begin
  with gsfontinfobn do
  begin
    FuncHdr.CommandCode := F_GRFXFONTINFOGET;
    FontNumber := num;
  end;
  MesaGrFunc(@gsfontinfobn);
  thefontinfo := gsfontinfobn.FontInfo;
end;

procedure GetTTYFontInfo(var thefontinfo : FontInfoRec);
{ Get font parameters of the current TTY font }
var gsfontinfo : GetSetFontInfoRec;
begin
  with gsfontinfo do
  begin
    FuncHdr.CommandCode := F_GRFXTFONTINFOGET;
  end;
  MesaGrFunc(@gsfontinfo);
  thefontinfo := gsfontinfo.FontInfo;
end;

procedure GetGraphicFontInfo(var thefontinfo : FontInfoRec);
{ Get font parameters of the current Graphic font }
var gsfontinfo : GetSetFontInfoRec;
begin
  with gsfontinfo do
  begin
    FuncHdr.CommandCode := F_GRFXNTFONTINFOGET;
  end;
  MesaGrFunc(@gsfontinfo);
  thefontinfo := gsfontinfo.FontInfo;
end;

procedure SetGraphicFont(thefontinfo : FontInfoRec);
{ Set graphic font parameters }
var gsfontinfo : GetSetFontInfoRec;
begin
  with gsfontinfo do
  begin
    FontInfo := thefontinfo;
    GState.GraphicFontInfo := gsfontinfo.FontInfo;  {shadow}
    FuncHdr.CommandCode := F_GRFXATTACHNTFONT;
  end;
  MesaGrFunc(@gsfontinfo);
end;

procedure SetTTYFont(thefontinfo : FontInfoRec);
{ Set TTY font parameters }
{ Note: this clears the current tty window! }
var gsfontinfo : GetSetFontInfoRec;
begin
  with gsfontinfo do
  begin
    FontInfo := thefontinfo;
    GState.CurTTYState.FontInfo := gsfontinfo.FontInfo;  {shadow}
    FuncHdr.CommandCode := F_GRFXATTACHFONT;
  end;
  MesaGrFunc(@gsfontinfo);
end;

procedure SetDefaultTTYFont(thefontinfo : FontInfoRec);
{ Set TTY font in GState only ... }
begin
  GState.CurTTYState.FontInfo := thefontinfo;  {shadow}
  GState.CurTTYState.CursorDimX := thefontinfo.CharWidth;
  GState.CurTTYState.CursorDimY := thefontinfo.Charheight;
  GState.CurTTYState.CursorOffsetY := 0;
  GState.CurTTYState.CursorOffsetX := 0;
end;

procedure SetGraphicFontToTTYFont;
{ This sets the Graphic font to be the same as the TTY font }
var fontinfo : FontInfoRec;
begin
  GetTTYFontInfo(fontinfo);
  SetGraphicFont(fontinfo);
end;

procedure SelectTTYFontByNumber(fn : byte);
{ Select TTY font from the two BIO fonts, clears the current window }
var thefontinfo : FontInfoRec;
begin
  case fn of
    0 : thefontinfo := GState.Font0Info;
    1 : thefontinfo := GState.Font1Info;
  end;
  SetTTYFont(thefontinfo);
end;

procedure SelectDefaultTTYFontByNumber(fn : byte);
{ Select TTY font from the two BIO fonts, clears the current window }
var thefontinfo : FontInfoRec;
begin
  case fn of
    0 : thefontinfo := GState.Font0Info;
    1 : thefontinfo := GState.Font1Info;
  end;
  SetDefaultTTYFont(thefontinfo);
end;

procedure SelectGraphicFontByNumber(fn : byte);
{ Select Graphic font from the two BIO fonts }
var thefontinfo : FontInfoRec;
begin
  case fn of
    0 : thefontinfo := GState.Font0Info;
    1 : thefontinfo := GState.Font1Info;
  end;
  SetGraphicFont(thefontinfo);
end;

procedure SetTTYWin(thewin : RectRec);
{ set new TTY window dimensions - clears new TTY window }
var
sttyrec : SetTTYWinRec;
gsttystaterec : GetSetTTYStateRec;
begin
  with sttyrec do
  begin
    FuncHdr.CommandCode :=F_GRFXSETTTYLOC;
    TTYWindow := thewin;
    SetTTYState(GState.CurTTYState,false);
  end;
  MesaGrFunc(@sttyrec);
  GetTTYState(GState.CurTTYState);
  { shadow }
end;

procedure SetTTYWinFull;
{ Set the TTY window to full screen size }
var
fullscreen : RectRec;
begin
  GetDispRect(fullscreen);
  SetTTYWin(fullscreen);
end;

procedure GBumout(s : string);
begin
  LCDOut;
  SetTTYWinFull;
  WakeCursor;
  writeln;
  writeln(chr(7),s);
  halt(2);
end;

procedure NBumout(s : string);
begin
  writeln;
  writeln(chr(7),s);
  halt(2);
end;

procedure SetClippedTTYWin(thewin : RectRec);
{ Center and Inset a TTY window inside thewin by quantizing the TTY window   }
{ to a multiple of character size. This prevents scrolling from leaving bits }
{ of previous lines on the screen.                                            }
var
extraspace : integer;
numoflines : integer;
topextra : integer;
ttywin: RectRec;
begin
  numoflines := (thewin.lr.y - thewin.ul.y) div GState.CurTTYState.FontInfo.CharHeight;
  extraspace := (thewin.lr.y - thewin.ul.y) - (numoflines * GState.CurTTYState.FontInfo.CharHeight);
  topextra := extraspace div 2;
  ttywin.UL.X := thewin.ul.x;
  ttywin.UL.Y := thewin.ul.y + topextra;
  ttywin.LR.X := thewin.lr.x;
  ttywin.LR.Y := thewin.lr.y - (extraspace - topextra);
  SetTTyWin(ttywin);
end;

procedure PrintDisplayInfo;
begin
  with GState.DisplayInfo do
  begin
    writeln('Display Info: ');
    writeln('DisplayDim: ',DisplayDim.X,' ',DisplayDim.Y);
    writeln('NybblesPerLine: ',NybblesPerLine);
    writeln('Cl1Start: ',Cl1Start);
    writeln('Cl1End: ',Cl1End);
    writeln('FLMStartOffset: ',FLMStartOffset);
    writeln('FLMEndOffset: ',FLMEndOffset);
    writeln('NumPlanes: ',NumPlanes);
    writeln('BaseParagraph: ',BaseParagraph);
    writeln('InvertDisp: ',InvertDisp);
  end;
end;

procedure InitGraphics;
{ Note that this routine does no real (screen) initialization other     }
{ than clearing the screen , setting the tty window to the full screen  }
{ size then turning off the cursor.                                     }
{ Its main purpose in life is to simplify and speed up graphics         }
{ procedures by shadowing some of the BIOS variables for quick access   }
{ and setting up reasonable values for foreground and background colors }
{ based on the invert screen (inv) boolean.                             }
var
gsdisplayinfo : GetSetDisplayInfoRec;
thefontinfo : FontInfoRec;
ttystate : TTYStateRec;
begin
  with GState do
  begin
    if MesaGrFuncsAvailQ then
    begin
      OldVideoDest := GetVidDest;
      GetKeySource(OldKeySource,OldMembToo);
      if MesaGrFuncsRev > 4 then
      begin
        LCDOut;
        GetDispRect(ScreenRect);
        GetTTYState(CurTTYState);
        OldTTYState := CurTTYState;
        GetTTYFontInfo(thefontinfo);
        CurTTYState.FontInfo := thefontinfo;
        GetGraphicFontInfo(thefontinfo);
        GraphicFontInfo := thefontinfo;
        GetBIOSFontInfo(thefontinfo,0);
        Font0Info := thefontinfo;
        GetBIOSFontInfo(thefontinfo,1);
        Font1Info := thefontinfo;
        if DisplayInfo.NumPlanes = 2 then PlaneMask := BothPlanes else PlaneMask := Plane1;
        SetTTYWinFull;
        SleepCursor;
        { chose display colors based on BIOS InvertDisplay boolean  }
        { we do this to simplfy mixing graphics with the TTY window }
        { procedures }
         if not DispInvertQ then
         begin
         { normal (black text on white bg) }
           BGColor := White;
           BGPat   := @SolidWhitePat;
           FGColor := Black;
           FGPat   := @SolidBlackPat;
           FGRasterOp := RasterOp_Rep;
           BGRasterOp := RasterOp_NRep;
         end
         else
         begin
         { normal (white text on black bg) }
           BGColor := Black;
           BGPat   := @SolidBlackPat;
           FGColor := White;
           FGPat   := @SolidWhitePat;
           FGRasterOp := RasterOp_NRep;
           BGRasterOp := RasterOp_Rep;
         end;
      end {if rev ok }
      else NBumout('BIOS Needs to be upgraded to run this code');
    end { if MesaGrFuncsAvailQ }
    else NBumout('Mesa graphics not supported on this computer!');
  end; { with gstate }
end;

procedure ExitGraphics;
begin
  with GState do
  begin
    SetVidDest(OldVideoDest);
    SetKeySource(OldKeySource,OldMembToo);
    If OldVideoDest = VIDDEST_LCD then
    begin
      { Should this be SetTTYWinFull?}
      SetTTYState(OldTTYState,true);
      WakeCursor;
    end;
  end;
end;

{ ***** BASIC drawing primitives that do not use GState ***** }
{ ***** and therefore have a lot of parameters...                   }

procedure BLT(sp: pointer;xdim,ydim,destx,desty : integer;ro,pm : byte);
{ Your basic BLT to screen function, used for drawing pictures and more... }
{ Parameters are sp          : Source pointer - pointer to source data     }
{                xdim,ydim   : dimensions of the source rectangle          }
{                desty,desty : target location on screen                   }
{                ro,pm       : the RasterOp and plane mask                 }
var
dont : boolean;
bltinfo : BLTRec;
begin
  with bltinfo do
  begin
    dont := false;
    FuncHdr.CommandCode := F_GRFXBITBLT;
    SourceRect.UL.X := 0;
    SourceRect.UL.Y := 0;
    SourceRect.LR.X := xdim-1;
    SourceRect.LR.Y := ydim-1;
    SourcePitch := (xdim+7) div 8;
    SourcePtr := sp;
    DestPoint.X := destx;
    DestPoint.Y := desty;
    { Note that we don't mess with the destination pitch! }

    { These patches do source clipping of objects if given negative         }
    { destination coordinates, but BLTed object would be partially visible. }
    { This is because the BIOS BLT routine dislikes negative coordinates!   }
    if destx < 0 then
    begin
      DestPoint.X := 0;
      if -destx >= xdim then dont := true;
      SourceRect.UL.X := -destx;
    end;
    if desty < 0 then
    begin
      DestPoint.Y := 0;
      if -desty >= ydim then dont := true;
      SourceRect.UL.Y := -desty;
    end;
    RasterOp := ro;
    PlaneMask := pm;
    if not dont then MesaGrFunc(@BLTInfo);
  end;
end;

procedure BLTWithOffset(sp: pointer;xsdim,ysdim,offx,offy : word; destrect : RectRec ;ro,pm : byte);
var
{ same as basic BLT but with offset into source rectangle }
{ to allow scrolling of image in smaller window }
bltinfo : BLTRec;
xddim,yddim : integer;
begin
  with bltinfo do
  begin
    xddim := destrect.LR.X-destrect.UL.X +1;
    yddim := destrect.LR.Y-destrect.UL.Y +1;
    FuncHdr.CommandCode := F_GRFXBITBLT;
    if offx+xddim >= xsdim-1 then offx := xsdim - xddim;
    if offy+yddim >= ysdim-1 then offy := ysdim - yddim;
    SourceRect.UL.X := offx;
    SourceRect.UL.Y := offy;
    SourceRect.LR.X := offx+xddim-1;
    SourceRect.LR.Y := offy+yddim-1;
    SourcePitch := (xsdim+7) div 8;
    SourcePtr := sp;
    DestPoint.X := DestRect.UL.X;
    DestPoint.Y := DestRect.UL.Y;
    { Note that we don't mess with the destination pitch! }
    RasterOp := ro;
    PlaneMask := pm;
    MesaGrFunc(@BLTInfo);
  end;
end;

procedure BltObj(pic : PictureRec; destx,desty : integer;ro,pm :byte);
begin
  BLT(pic.PicPtr,pic.XDim,pic.YDim,destx,desty,ro,pm);
end;

procedure BltObjWithOffset(pic : PictureRec; offsx,offsy : integer;destrect : RectRec;ro,pm :byte);
begin
  BLTWithOffset(pic.PicPtr,pic.XDim,pic.YDim,offsx,offsy,destrect,ro,pm);
end;

procedure UnBLT(sp: pointer;xdim,ydim,srcx,srcy : integer;ro,pm : byte);
{ your basic BLT from screen function }
{ sp must point to a buffer with enough space to save the region }
{ the size of the required buffer can be determined }
var
unbltinfo : UnBLTRec;
begin
  with unbltinfo do
  begin
    FuncHdr.CommandCode := F_GRFXUnBITBLT;
    DestRect.UL.X := 0;
    DestRect.UL.Y := 0;
    DestRect.LR.X := xdim-1;
    DestRect.LR.Y := ydim-1;
    DestPitch := (xdim+7) div 8;
    DestPtr := sp;
    SourcePoint.X := srcx;
    SourcePoint.Y := srcy;
    RasterOp := ro;
    PlaneMask := pm;
    MesaGrFunc(@unbltinfo);
  end;
end;

procedure DrawLine(startp,endp: pointRec;color,ro,pm : byte);
var drawlrec : DrawLineRec;
begin
  with drawlrec do
  begin
    FuncHdr.CommandCode :=F_GRFXDRAWLINE;
    StartPoint := startp;
    EndPoint := endp;
    LineColor := color;
    SkipFirst := 0;    { no }
    SkipLast := 0;     { no }
    RasterOp := ro;
    PlaneMask := pm;
  end;
  MesaGrFunc(@drawlrec);
end;

procedure DrawRectangle(therect : RectRec; color,ro,pm : byte);
var p1,p2 : pointRec;
begin
  with therect do
  begin
    p1.x := UL.X;
    p1.y := UL.Y;
    p2.x := LR.X;
    p2.y := UL.Y;
    DrawLine(p1,p2,color,ro,pm);
    p1.x := LR.X;
    p1.y := UL.Y;
    p2.x := LR.X;
    p2.y := LR.Y;
    DrawLine(p1,p2,color,ro,pm);
    p1.x := LR.X;
    p1.y := LR.Y;
    p2.x := UL.X;
    p2.y := LR.Y;
    DrawLine(p1,p2,color,ro,pm);
    p1.x := UL.X;
    p1.y := LR.Y;
    p2.x := UL.X;
    p2.y := UL.Y;
    DrawLine(p1,p2,color,ro,pm);
  end;
end;

procedure DrawChar(destx,desty,ch : word;ro,pm : byte);
{ Draws a character from the non-tty (alternate) font to anywhere }
{ on the screen }
var drawcrec : DrawCharRec;
begin
  with drawcrec do
  begin
    FuncHdr.CommandCode := F_GRFXDRAWCHAR;
    DestPoint.x := destx;
    DestPoint.y := desty;
    TheChar := ch;
    RasterOp := ro;
    PlaneMask := pm;
  end;
  MesaGrFunc(@drawcrec);
end;

procedure DrawString(destx,desty,incx,incy : word;st : string;ro,pm : byte);
var
xpos,ypos : word;
index: byte;
begin
  xpos := destx;
  ypos := desty;
  for index := 1 to length(st) do
  begin
    DrawChar(xpos,ypos,word(st[index]),ro,pm);
    xpos := xpos + incx;
    ypos := ypos + incy;
  end;
end;

{ ***** More complex drawing primitives that use GState to ***** }
{ ***** reduce the number of parameters passed from call to call ***** }

procedure DrawCenteredStringInBox(box : RectRec; st : string;ro,pm : byte);
var
stwidth,destx,desty : word;
begin
  stwidth := GState.GraphicFontInfo.CharWidth * length(st);
{  if (RectXSize(box) >= stwidth) and (RectYSize(box) >= GState.GraphicFontInfo.CharHeight) then }
  begin
    destx := box.UL.X + ((RectXSize(box) - stwidth) div 2) + 1;
    desty := box.UL.Y + (RectYSize(box) - GState.GraphicFontInfo.CharHeight) div 2;
    DrawString(destx,desty,GState.GraphicFontInfo.CharWidth,0,st,ro,pm);
  end;
end;

procedure DrawCenteredStringInBoxVertically(box : RectRec; st : string;ro,pm : byte);
var
stheight,destx,desty : word;
begin
  stheight := GState.GraphicFontInfo.CharHeight * length(st);
{  if (RectXSize(box) >= GState.GraphicFontInfo.CharWidth) and (RectYSize(box) >= stheight) then }
  begin
    destx := box.UL.X + (RectXSize(box) - GState.GraphicFontInfo.CharWidth) div 2;
    desty := box.UL.Y + (RectYSize(box) - stheight) div 2 +1;
    DrawString(destx,desty,0,GState.GraphicFontInfo.CharHeight,st,ro,pm);
  end;
end;

procedure DrawLeftJustifiedStringInBox(box : RectRec; st : string;margin,ro,pm : byte);
var
stwidth,destx,desty : word;
begin
  stwidth := GState.GraphicFontInfo.CharWidth * length(st);
{  if (RectXSize(box) >= stwidth+margin) and (RectYSize(box) >= GState.GraphicFontInfo.CharHeight) then}
  begin
    destx := box.UL.X + margin;
    desty := box.UL.Y + (RectYSize(box) - GState.GraphicFontInfo.CharHeight) div 2 + 1;
    DrawString(destx,desty,GState.GraphicFontInfo.CharWidth,0,st,ro,pm);
  end;
end;

procedure DrawTopJustifiedStringInBoxVertically(box : RectRec; st : string;margin,ro,pm : byte);
var
stheight,destx,desty : word;
begin
  stheight := GState.GraphicFontInfo.CharHeight * length(st);
{  if (RectXSize(box) >= GState.GraphicFontInfo.CharWidth) and (RectYSize(box) >= stheight) then}
  begin
    destx := box.UL.X + (RectXSize(box) - GState.GraphicFontInfo.CharWidth) div 2;
    desty := box.UL.Y + margin;
    DrawString(destx,desty,0,GState.GraphicFontInfo.CharHeight,st,ro,pm);
  end;
end;

procedure DrawCenteredStringAtPoint(pt : PointRec; st : string;ro,pm : byte);
var
stwidth,destx,desty : word;
begin
  stwidth := GState.GraphicFontInfo.CharWidth * length(st);
  begin
    destx := pt.X - (stwidth div 2) + 1;
    desty := pt.Y - (GState.GraphicFontInfo.CharHeight div 2);
    DrawString(destx,desty,GState.GraphicFontInfo.CharWidth,0,st,ro,pm);
  end;
end;

procedure DrawCenteredStringAtPointVertically(pt : PointRec; st : string;ro,pm : byte);
var
stheight,destx,desty : word;
begin
  stheight := GState.GraphicFontInfo.CharHeight * length(st);
  begin
    destx := pt.X - (GState.GraphicFontInfo.CharWidth div 2);
    desty := pt.Y - (stheight div 2) + 1;
    DrawString(destx,desty,0,GState.GraphicFontInfo.CharHeight,st,ro,pm);
  end;
end;

procedure DrawLeftJustifiedStringAtPoint(pt : PointRec; st : string;margin,ro,pm : byte);
var
stwidth,destx,desty : word;
begin
  stwidth := GState.GraphicFontInfo.CharWidth * length(st);
  begin
    destx := pt.X + margin;
    desty := pt.Y - (GState.GraphicFontInfo.CharHeight div 2) + 1;
    DrawString(destx,desty,GState.GraphicFontInfo.CharWidth,0,st,ro,pm);
  end;
end;

procedure DrawTopJustifiedStringAtPointVertically(pt : PointRec; st : string;margin,ro,pm : byte);
var
stheight,destx,desty : word;
begin
  stheight := GState.GraphicFontInfo.CharHeight * length(st);
  begin
    destx := pt.X - (GState.GraphicFontInfo.CharWidth div 2) + 1;
    desty := pt.Y + margin;
    DrawString(destx,desty,0,GState.GraphicFontInfo.CharHeight,st,ro,pm);
  end;
end;

procedure GWaitForKeyPress;
begin
  SetVidDest(VIDDEST_STUB);
  while not QKeyPressed do;
  SetVidDest(VIDDEST_LCD);
end;

procedure SystemErrorNotifier(message1,message2 : string);
var
errbox,textbox,textboxt,textboxb : RectRec;
fookey,bufsize : word;
bufptr : pointer;
begin
  textbox.UL.X := CenterOfRectX(GState.ScreenRect)-(UnHappyRec.XDIM*3 div 2);
  textbox.UL.Y := CenterOfRectY(GState.ScreenRect)-(UnHappyRec.YDIM   div 2);
  textbox.LR.X := textbox.UL.X + UnHappyRec.XDim *3 -1;
  textbox.LR.Y := textbox.UL.Y + UnHappyRec.YDim    -1;
  InsetRect(textbox,errbox,4,4);
  BufSize := GetRegionBuffSize(errbox);
  GetMem(bufptr,bufsize);
  SaveRegion(errbox,bufptr);
  FillRegion(errbox,Gstate.FGPat,RasterOp_Rep,BothPlanes);
  FillRegion(textBox,Gstate.BGPat,RasterOp_Rep,BothPlanes);
  BLTObj(UnhappyRec,textbox.UL.X,textbox.UL.Y,RasterOp_Rep,Plane1);
  InsetEdge(textbox,textbox,left,-UnHappyXDim);
  SplitRectY(textbox,textboxt,textboxb,RectYSize(textbox)div 2);
  SelectGraphicFontByNumber(0);
  DrawCenteredStringInBox(textboxt,message1,GState.FGRasterOp,BothPlanes);
  DrawCenteredStringInBox(textboxb,message2,GState.FGRasterOp,BothPlanes);
  GWaitForKeyPress;
  fookey := QKeyRead;
  RestoreRegion(bufptr);
  FreeMem(bufptr,bufsize);
end;

procedure OneLineLabeledButton(box : RectRec;st : string);
begin
  FillRegion(box,Gstate.FGPat,RasterOp_Rep,BothPlanes);
  DrawCenteredStringInBox(box,st,GState.BGRasterOp,GState.PlaneMask);
end;

procedure TwoLineLabeledButton(box : RectRec;st1,st2 : string);
var tbox,bbox : Rectrec;
begin
  FillRegion(box,Gstate.FGPat,RasterOp_Rep,BothPlanes);
  SplitRectY(box,tbox,bbox,RectXSize(box) div 2);
  DrawCenteredStringInBox(tbox,st1,GState.BGRasterOp,GState.PlaneMask);
  DrawCenteredStringInBox(bbox,st2,GState.BGRasterOp,GState.PlaneMask);
end;

procedure HorBarGraphPlace(var bg : BarGraphRec;size : RectRec;min,max,ticint : integer; doout :boolean);
{ Draw a horizontal bar graph at the rectangle size. Min and max are the    }
{ minimum and maximum full scale input values expected (they are integers)  }
{ ticint is the interval between tic marks. If ticint is 0 no ticmarks are  }
{ drawn. If the doout boolean is true, the outline is drawn otherwise not.  }
var
thetic : integer;
ticcount : word;
ticp1,ticp2,ticp3,ticp4 : PointRec;
ticscale : real;
begin
  with bg do
  begin
    Outline := size;
    FillRegion(size,GState.BGPat,RasterOp_Rep,BothPlanes);
    CurBar := Outline;
    if doout then
    begin
      DrawRectangle(Outline,GState.FGColor,GState.FGRasterOp,BothPlanes);
      InsetRect(OutLine,CurBar,-1,-1);
    end;
    if ticint <> 0 then
    begin
      InsetRect(CurBar,CurBar,0,-2);
    end;
    ScaleInfo.ScaleRect := Outline;
    ScaleInfo.Scale := RectXSize(CurBar)/(max-min);
    ScaleInfo.Offset := -min;
    ScaleInfo.MinReading := min;
    ScaleInfo.MaxReading := max;
    ScaleInfo.TicInterval := ticint;
    ScaleInfo.MeterType := Horizontal;
    if ticint <> 0 then
    begin
      { draw the ticmarks }
      ticp1.y := Outline.UL.Y;
      ticp2.y := CurBar.UL.Y -1;
      ticp3.y := Outline.LR.Y;
      ticp4.y := CurBar.LR.Y +1;
      thetic := min;
      while thetic <= max do
      begin
        ticp1.X := ScaleInfo.ScaleRect.UL.X + trunc(((thetic+ScaleInfo.Offset) * ScaleInfo.Scale) + 0.5);
        ticp2.X := ticp1.X;
        ticp3.X := ticp1.X;
        ticp4.X := ticp1.X;
        DrawLine(ticp1,ticp2,GState.FGColor,GState.FGRasterOp,BothPlanes);
        DrawLine(ticp3,ticp4,GState.FGColor,GState.FGRasterOp,BothPlanes);
        ticp1.X := ticp1.X +1;
        ticp2.X := ticp1.X;
        ticp3.X := ticp1.X;
        ticp4.X := ticp1.X;
        DrawLine(ticp1,ticp2,GState.FGColor,GState.FGRasterOp,BothPlanes);
        DrawLine(ticp3,ticp4,GState.FGColor,GState.FGRasterOp,BothPlanes);
        thetic := thetic + ticint;
      end;
    end;
    BarLength := 0;
    CurBar.LR.X := CurBar.UL.X + BarLength;
    DeltaBar := CurBar;
    { make 0 length current bar }
    Seal := ValidSeal;
  end;
end;

procedure VertBarGraphPlace(var bg : BarGraphRec;size : RectRec;min,max,ticint : integer; doout :boolean);
{ Draw a vertical bar graph at the rectangle size. Min and max are the      }
{ minimum and maximum full scale input values expected (they are integers)  }
{ ticint is the interval between tic marks. If ticint is 0 no ticmarks are  }
{ drawn. If the doout boolean is true, the outline is drawn otherwise not.  }
var
thetic : integer;
ticcount : word;
ticp1,ticp2,ticp3,ticp4 : PointRec;
ticscale : real;
begin
  with bg do
  begin
    Outline := size;
    FillRegion(size,GState.BGPat,RasterOp_Rep,BothPlanes);
    CurBar := Outline;
    if doout then
    begin
      DrawRectangle(Outline,GState.FGColor,GState.FGRasterOp,BothPlanes);
      InsetRect(OutLine,CurBar,-1,-1);
    end;
    if ticint <> 0 then
    begin
      InsetRect(CurBar,CurBar,-2,-0);
    end;
    ScaleInfo.ScaleRect := Outline;
    ScaleInfo.Scale := RectYSize(CurBar)/(max-min);
    ScaleInfo.Offset := -min;
    ScaleInfo.MinReading := min;
    ScaleInfo.MaxReading := max;
    ScaleInfo.TicInterval := ticint;
    ScaleInfo.MeterType := Vertical;
    if ticint <> 0 then
    begin
      { draw the ticmarks }
      ticp1.X := Outline.UL.X;
      ticp2.X := CurBar.UL.X -1;
      ticp3.X := Outline.LR.X;
      ticp4.X := CurBar.LR.X +1;
      thetic := min;
      while thetic <= max do
      begin
        ticp1.Y := ScaleInfo.ScaleRect.LR.Y - trunc(((thetic+ScaleInfo.Offset) * ScaleInfo.Scale) + 0.5);
        ticp2.Y := ticp1.Y;
        ticp3.Y := ticp1.Y;
        ticp4.Y := ticp1.Y;
        DrawLine(ticp1,ticp2,GState.FGColor,GState.FGRasterOp,BothPlanes);
        DrawLine(ticp3,ticp4,GState.FGColor,GState.FGRasterOp,BothPlanes);
        ticp1.Y := ticp1.Y -1;
        ticp2.Y := ticp1.Y;
        ticp3.Y := ticp1.Y;
        ticp4.Y := ticp1.Y;
        DrawLine(ticp1,ticp2,GState.FGColor,GState.FGRasterOp,BothPlanes);
        DrawLine(ticp3,ticp4,GState.FGColor,GState.FGRasterOp,BothPlanes);
        thetic := thetic + ticint;
      end;
    end;
    BarLength := 0;
    CurBar.UL.Y := CurBar.LR.Y - BarLength;
    DeltaBar := CurBar;
    { make 0 length current bar }
    Seal := ValidSeal;
  end;
end;

function HorBarGraphCreate(size : RectRec;min,max,ticint : integer;doout :boolean): BarGraphPtr;
var bg : BarGraphPtr;
begin
  GetMem(bg,SizeOf(BarGraphRec));
  with bg^ do
  begin
    BufSize := GetRegionBuffSize(size);
    GetMem(BufPtr,BufSize);
    SaveRegion(size,BufPtr);
    HorBarGraphPlace(bg^,size,min,max,ticint,doout);
    HorBarGraphCreate := bg;
  end;
end;

function VertBarGraphCreate(size : RectRec;min,max,ticint : integer;doout :boolean): BarGraphPtr;
var bg : BarGraphPtr;
begin
  GetMem(bg,SizeOf(BarGraphRec));
  with bg^ do
  begin
    BufSize := GetRegionBuffSize(size);
    GetMem(BufPtr,BufSize);
    SaveRegion(size,BufPtr);
    VertBarGraphPlace(bg^,size,min,max,ticint,doout);
    VertBarGraphCreate := bg;
  end;
end;

procedure HorBarGraphUpdate(bg : BarGraphPtr; nv : integer);
var
newbarlength : word;
deltabar : RectRec;
begin
  with bg^ do
  begin
    newbarlength := trunc((nv+ScaleInfo.Offset) * ScaleInfo.Scale +0.5);
    if newbarlength > BarLength then
    begin
      DeltaBar.UL.X := CurBar.LR.X;
      DeltaBar.LR.X := CurBar.UL.X + newbarlength -1;
      FillRegion(DeltaBar,GState.FGPat,RasterOp_Rep,BothPlanes);
      CurBar.LR.X   := CurBar.UL.X + newbarlength -1;
      BarLength := newbarlength;
    end;
    if newbarlength < BarLength then
    begin
      DeltaBar.UL.X := CurBar.UL.X + newbarlength;
      DeltaBar.LR.X := CurBar.LR.X;
      FillRegion(DeltaBar,GState.BGPat,RasterOp_Rep,BothPlanes);
      CurBar.LR.X   := CurBar.UL.X + newbarlength;
      BarLength := newbarlength;
    end;
  end;
end;

procedure VertBarGraphUpdate(bg : BarGraphPtr; nv : integer);
var
newbarlength : word;
deltabar : RectRec;
begin
  with bg^ do
  begin
    newbarlength := trunc((nv+ScaleInfo.Offset) * ScaleInfo.Scale +0.5);
    if newbarlength > BarLength then
    begin
      DeltaBar.LR.Y := CurBar.UL.Y;
      DeltaBar.UL.Y := CurBar.LR.Y - newbarlength +1;
      FillRegion(DeltaBar,GState.FGPat,RasterOp_Rep,BothPlanes);
      CurBar.UL.Y   := CurBar.LR.Y - newbarlength +1;
      BarLength := newbarlength;
    end;
    if newbarlength < BarLength then
    begin
      DeltaBar.UL.Y := CurBar.UL.Y;
      DeltaBar.LR.Y := CurBar.LR.Y - newbarlength;
      FillRegion(DeltaBar,GState.BGPat,RasterOp_Rep,BothPlanes);
      CurBar.UL.Y   := CurBar.LR.Y - newbarlength;
      BarLength := newbarlength;
    end;
  end;
end;

procedure BarGraphUpdate(bg : BarGraphPtr; nv : integer);
begin
  with bg^ do
  begin
    if Seal = ValidSeal then
    begin
      case ScaleInfo.MeterType of
        Vertical   : VertBarGraphUpdate(bg,nv);
        Horizontal : HorBarGraphUpdate(bg,nv)
      end;
    end
    else SystemErrorNotifier('Bargrph:','No-init');
  end;
end;

procedure BarGraphDestroy(bg : BarGraphPtr);
begin
  with bg^ do
  begin
    if Seal = ValidSeal then
    begin
      Seal := InvalidSeal;
      RestoreRegion(BufPtr);
      FreeMem(BufPtr,BufSize);
      FreeMem(bg,SizeOf(BarGraphRec));
    end
    else SystemErrorNotifier('Bargrph:','No-init');
  end;
end;

procedure VertEdgeMeterPlace(var em :EdgeMeterRec; size:RectRec;min,max: integer;ticint : word);
var
thetic : integer;
ticcount : word;
ticp1,ticp2 :PointRec;
begin
  with em do
  begin
    Outline := size;
    FillRegion(OutLine,GState.BGPat,RasterOp_Rep,BothPlanes);
    { create the scale rectangle and the pointer range rectangle }
    SplitRectX(Outline,PtrRangeRect,ScaleInfo.ScaleRect,(RectXSize(OutLine) div 2));
    { shrink the vertical range to accomodate pointer size }
    InsetRect(PtrRangeRect,PtrRangeRect,0,-(RectXSize(OutLine) div 2));
    InsetRect(ScaleInfo.ScaleRect,ScaleInfo.ScaleRect,0,-(RectXSize(OutLine) div 2));
    ScaleInfo.ScaleRect.UL.X := ScaleInfo.ScaleRect.UL.X +1; { avoid tip of pointer }
    { now save information for labeling meter }
    ScaleInfo.MinReading := min;
    ScaleInfo.MaxReading := max;
    ScaleInfo.TicInterval := ticint;
    ScaleInfo.MeterType := Vertical;
    ScaleInfo.Scale := (RectYSize(ScaleInfo.ScaleRect)-1)/(max-min);
    ScaleInfo.Offset := -min;
    PrevData := min;
    DrawRectangle(ScaleInfo.ScaleRect,GState.FGColor,GState.FGRasterOp,BothPlanes);
    if ticint <> 0 then
    begin
      { draw the ticmarks }
      ticp1.X := ScaleInfo.ScaleRect.UL.X+1;
      ticp2.X := ScaleInfo.ScaleRect.LR.X-1;
      thetic := min;
      while thetic <= max do
      begin
        ticp1.Y := ScaleInfo.ScaleRect.LR.Y - trunc(((thetic+ScaleInfo.Offset) * ScaleInfo.Scale) + 0.5);
        ticp2.Y := ticp1.Y;
        DrawLine(ticp1,ticp2,GState.FGColor,GState.FGRasterOp,BothPlanes);
        thetic := thetic + ticint;
      end;
    end;
    { Now we scale our pointer ...  This math assumes 90 deg pointer! }
    PtrSize := RectXSize(PtrRangeRect);
    BltOffset := PtrRightXdim-PtrSize;
    if BltOffset < 0 then BltOffset := 0;
    PtrRect := PtrRangeRect;
    { These off by one tweaks depend on the specific pointer bitmap   }
    { they (and the similar one in VertEdgeMeterUpdate may need to be }
    { re-tweaked if the pointer bitmap is changed!     -->            }
    PtrRect.UL.Y := PtrRangeRect.LR.Y -PtrSize +1;
    PtrRect.LR.Y := PtrRangeRect.LR.Y +PtrSize +1;
    { draw the initial pointer }
    BltWithOffset(@PtrRight,PtrRightXDim,PtrRightYDim,BltOffset,BltOffset,PtrRect,RasterOp_Xor,BothPlanes);
  end;
end;

function VertEdgeMeterCreate(size:RectRec;min,max: integer;ticint : word) :EdgeMeterPtr;
var em : EdgeMeterPtr;
begin
  GetMem(em,SizeOf(edgeMeterRec));
  with em^ do
  begin
    BufSize := GetRegionBuffSize(size);
    GetMem(BufPtr,BufSize);
    SaveRegion(size,BufPtr);
    VertEdgeMeterPlace(em^,size,min,max,ticint);
    VertEdgeMeterCreate := em;
    Seal := ValidSeal;
  end;
end;

procedure HorEdgeMeterPlace(var em :EdgeMeterRec; size:RectRec;min,max: integer;ticint : word);
var
thetic : integer;
ticcount : word;
ticp1,ticp2 :PointRec;
begin
  with em do
  begin
    Outline := size;
    FillRegion(OutLine,GState.BGPat,RasterOp_Rep,BothPlanes);
    { create the scale rectangle and the pointer range rectangle }
    SplitRectY(Outline,PtrRangeRect,ScaleInfo.ScaleRect,(RectYSize(OutLine) div 2));
    { shrink the horizontal range to accomodate pointer size }
    InsetRect(PtrRangeRect,PtrRangeRect,-(RectYSize(OutLine) div 2),0);
    InsetRect(ScaleInfo.ScaleRect,ScaleInfo.ScaleRect,-(RectYSize(OutLine) div 2),0);
    ScaleInfo.ScaleRect.UL.Y := ScaleInfo.ScaleRect.UL.Y +1; { avoid tip of pointer }
    { now save information for labeling meter }
    ScaleInfo.MinReading := min;
    ScaleInfo.MaxReading := max;
    ScaleInfo.TicInterval := ticint;
    ScaleInfo.MeterType := Horizontal;
    ScaleInfo.Scale := (RectXSize(ScaleInfo.ScaleRect)-1)/(max-min);
    ScaleInfo.Offset := -min;
    PrevData := min;
    DrawRectangle(ScaleInfo.ScaleRect,GState.FGColor,GState.FGRasterOp,BothPlanes);
    if ticint <> 0 then
    begin
      { draw the ticmarks }
      ticp1.Y := ScaleInfo.ScaleRect.UL.Y+1;
      ticp2.Y := ScaleInfo.ScaleRect.LR.Y-1;
      thetic := min;
      while thetic <= max do
      begin
        ticp1.X := ScaleInfo.ScaleRect.UL.X + trunc(((thetic+ScaleInfo.Offset) * ScaleInfo.Scale) + 0.5);
        ticp2.X := ticp1.X;
        DrawLine(ticp1,ticp2,GState.FGColor,GState.FGRasterOp,BothPlanes);
        thetic := thetic + ticint;
      end;
    end;
    { Now we scale our pointer ...  This math assumes 90 deg pointer! }
    PtrSize := RectYSize(PtrRangeRect);
    BltOffset := PtrDownYdim-PtrSize;
    if BltOffset < 0 then BltOffset := 0;
    PtrRect := PtrRangeRect;
    { These off by one tweaks depend on the specific pointer bitmap   }
    { they (and the similar one in HorEdgeMeterUpdate may need to be  }
    { re-tweaked if the pointer bitmap is changed!     -->            }
    PtrRect.UL.X := PtrRangeRect.UL.X -PtrSize -1;
    PtrRect.LR.X := PtrRangeRect.UL.X +PtrSize -1;
    { draw the initial pointer }
    BltWithOffset(@PtrDown,PtrDownXDim,PtrDownYDim,BltOffset,BltOffset,PtrRect,RasterOp_Xor,BothPlanes);
  end;
end;

function HorEdgeMeterCreate(size:RectRec;min,max: integer;ticint : word) :EdgeMeterPtr;
var
em : EdgeMeterPtr;
begin
  GetMem(em,SizeOf(EdgeMeterRec));
  with em^ do
  begin
    BufSize := GetRegionBuffSize(size);
    GetMem(BufPtr,BufSize);
    SaveRegion(size,BufPtr);
    HorEdgeMeterPlace(em^,size,min,max,ticint);
    HorEdgeMeterCreate := em;
    Seal := ValidSeal;
  end;
end;

procedure VertEdgeMeterUpdate(em :EdgeMeterPtr;newdata : integer);
var
newpos : word;
begin
  with em^ do
  begin
    if Seal = ValidSeal then
    begin
      newpos := ScaleInfo.ScaleRect.LR.Y - trunc((newdata+ScaleInfo.Offset) * ScaleInfo.Scale +0.5)+1;
      { only move pointer if necessary }
      if (newpos - PtrSize) <> PtrRect.UL.Y then
      begin
        { first undraw old pointer }
        BltWithOffset(@PtrRight,PtrRightXDim,PtrRightYDim,BltOffset,BltOffset,PtrRect,RasterOp_Xor,BothPlanes);
        { move the pointer position }
        PtrRect.UL.Y := newpos - PtrSize;
        PtrRect.LR.Y := newpos + PtrSize;
        { then redraw new pointer }
        BltWithOffset(@PtrRight,PtrRightXDim,PtrRightYDim,BltOffset,BltOffset,PtrRect,RasterOp_Xor,BothPlanes);
      end;
    end
    else SystemErrorNotifier('Vertmet:','No-init');
  end;
end;

procedure HorEdgeMeterUpdate(em : EdgeMeterPtr;newdata : integer);
var
newpos : word;
begin
  with em^ do
  begin
    if Seal = ValidSeal then
    begin
      newpos := ScaleInfo.ScaleRect.UL.X + trunc((newdata+ScaleInfo.Offset) * ScaleInfo.Scale +0.5) -1;
      { only move pointer if necessary }
      if (newpos - PtrSize) <> PtrRect.UL.X then
      begin
        { first undraw old pointer }
        BltWithOffset(@PtrDown,PtrDownXDim,PtrDownYDim,BltOffset,BltOffset,PtrRect,RasterOp_Xor,BothPlanes);
        { move the pointer position }
        PtrRect.UL.X := newpos - PtrSize;
        PtrRect.LR.X := newpos + PtrSize;
        { then redraw new pointer }
        BltWithOffset(@PtrDown,PtrDownXDim,PtrDownYDim,BltOffset,BltOffset,PtrRect,RasterOp_Xor,BothPlanes);
      end;
    end
    else SystemErrorNotifier('Vertmet:','No-init');
  end;
end;

procedure EdgeMeterDestroy(em : EdgeMeterPtr);
begin
  with em^ do
  begin
    if Seal = ValidSeal then
    begin
      Seal := InvalidSeal;
      { remove pointer }
      BltWithOffset(@PtrRight,PtrRightXDim,PtrRightYDim,BltOffset,BltOffset,PtrRect,RasterOp_Xor,BothPlanes);
      RestoreRegion(BufPtr);
      FreeMem(BufPtr,BufSize);
      FreeMem(em,SizeOf(EdgeMeterRec));
    end
    else SystemErrorNotifier('Edgemet:','No-init');
  end;
end;

procedure LabelEdgeMeter(em : EdgeMeterPtr; tic : integer; mlabel : string);
var
labelrect : RectRec;
labelcent : word;
begin
  { Label meter scale with normal text on right for vertical meters     }
  { and vertical text for horizontal meters.                            }
  { Probably should add capability of alternate text orientations.      }
  { Call with meter to label, which tic mark to label, and label string.}
  with em^ do
  begin
    if Seal = ValidSeal then
    begin
      with ScaleInfo do
      begin
        case MeterType of
          Vertical :
          begin
            labelcent := ScaleRect.LR.Y - trunc(((tic*TicInterval) * Scale) + 0.5);
            labelrect.UL.X := ScaleRect.LR.X;
            labelrect.UL.Y := labelcent - (Gstate.GraphicFontInfo.CharHeight div 2);
            labelrect.LR.X := ScaleRect.LR.X + (GState.GraphicFontInfo.CharWidth * length(mlabel));
            labelrect.LR.Y := LabelRect.UL.Y + Gstate.GraphicFontInfo.CharHeight;
            DrawLeftJustifiedStringInBox(labelrect,mlabel,2,GState.FGRasterOp,BothPlanes)
          end;
          Horizontal :
          begin
            labelcent := ScaleRect.UL.X + trunc(((tic*TicInterval) * Scale) + 0.5);
            labelrect.UL.Y := ScaleRect.LR.Y;
            labelrect.UL.X := labelcent - (Gstate.GraphicFontInfo.CharWidth div 2);
            labelrect.LR.Y := ScaleRect.LR.Y + (GState.GraphicFontInfo.CharHeight * length(mlabel));
            labelrect.LR.X := LabelRect.UL.X + Gstate.GraphicFontInfo.CharWidth;
            DrawTopJustifiedStringInBoxVertically(labelrect,mlabel,2,GState.FGRasterOp,BothPlanes)
          end;
        end;
      end;
    end
    else SystemErrorNotifier('Edgemet:','No-init');
  end;
end;

{ Removed Scaleinfo.Offset from all label-meter procedures          }
{ (since tic identifiers are always 0 through ntics. Offset caused  }
{ problems with meters with non zero minimum readings) PCW 11-22-95 }

procedure LabelBarGraph(bg : BarGraphPtr; tic : integer; mlabel : string);
var
labelrect : RectRec;
labelcent : word;
begin
  { Label meter scale with normal text on right for vertical meters }
  { and vertical text for horizontal meters.                        }
  { Probably should add capability of alternate text orientations.  }
  { Call with pointer to meter to label, which tic mark to label,   }
  { and label string.                                               }
  with bg^.ScaleInfo do
  begin
    if bg^.Seal = ValidSeal then
    begin
      case MeterType of
        Vertical :
        begin
          labelcent := ScaleRect.LR.Y - trunc(((tic*TicInterval) * Scale) + 0.5);
          labelrect.UL.X := ScaleRect.LR.X;
          labelrect.UL.Y := labelcent - (Gstate.GraphicFontInfo.CharHeight div 2);
          labelrect.LR.X := ScaleRect.LR.X + (GState.GraphicFontInfo.CharWidth * length(mlabel));
          labelrect.LR.Y := LabelRect.UL.Y + Gstate.GraphicFontInfo.CharHeight;
          DrawLeftJustifiedStringInBox(labelrect,mlabel,2,GState.FGRasterOp,BothPlanes)
        end;
        Horizontal :
        begin
          labelcent := ScaleRect.UL.X + trunc(((tic*TicInterval) * Scale) + 0.5);
          labelrect.UL.Y := ScaleRect.LR.Y;
          labelrect.UL.X := labelcent - (Gstate.GraphicFontInfo.CharWidth div 2);
          labelrect.LR.Y := ScaleRect.LR.Y + (GState.GraphicFontInfo.CharHeight * length(mlabel));
          labelrect.LR.X := LabelRect.UL.X + Gstate.GraphicFontInfo.CharWidth;
          DrawTopJustifiedStringInBoxVertically(labelrect,mlabel,2,GState.FGRasterOp,BothPlanes)
        end;
      end;
    end
    else SystemErrorNotifier('Bargrph:','No-init');
  end;
end;

procedure TitleRect(tr : RectRec; tl : Edge; tp : TextPos; title : string);
{ Places a title string outside of the supplied rectangle }
{ Top and bottom locations use horizontal text            }
{ Left and right locations use vertical text              }
var
textloc : PointRec;
begin
  case tl of
    Top :
    begin
      textloc.Y := tr.UL.Y - (GState.GraphicFontInfo.CharHeight div 2) - 2;
      case tp of
        LeftJustify   :
        begin
          textloc.X := tr.UL.X;
          DrawLeftJustifiedStringAtPoint(textloc,title,2,GState.FGRasterOp,BothPlanes);
        end;
        CenterJustify :
        begin
          textloc.X := (tr.UL.X + tr.LR.X) div 2;
          DrawCenteredStringAtPoint(textloc,title,GState.FGRasterOp,BothPlanes);
        end;
      end;
    end;
    Bottom :
    begin
      textloc.Y := tr.LR.Y + (GState.GraphicFontInfo.CharHeight div 2) + 2;
      case tp of
        LeftJustify   :
        begin
          textloc.X := tr.UL.X;
          DrawLeftJustifiedStringAtPoint(textloc,title,2,GState.FGRasterOp,BothPlanes);
        end;
        CenterJustify :
        begin
          textloc.X := (tr.UL.X + tr.LR.X) div 2;
          DrawCenteredStringAtPoint(textloc,title,GState.FGRasterOp,BothPlanes);
        end;
      end;
    end;
    Left :
    begin
      textloc.X := tr.UL.X - (GState.GraphicFontInfo.CharWidth div 2) -2;
      case tp of
        TopJustify    :
        begin
          textloc.Y := tr.UL.Y + 2;
          DrawTopJustifiedStringAtPointVertically(textloc,title,2,GState.FGRasterOp,BothPlanes);
        end;
        CenterJustify :
        begin
          textloc.Y := (tr.UL.Y + tr.LR.Y) div 2;
          DrawCenteredStringAtPointVertically(textloc,title,GState.FGRasterOp,BothPlanes);
        end;
      end;
    end;
    Right :
    begin
      textloc.X := tr.LR.X + (GState.GraphicFontInfo.CharWidth div 2) +2;
      case tp of
        TopJustify    :
        begin
          textloc.Y := tr.UL.Y + 2;
          DrawTopJustifiedStringAtPointVertically(textloc,title,2,GState.FGRasterOp,BothPlanes);
        end;
        CenterJustify :
        begin
          textloc.Y := (tr.UL.Y + tr.LR.Y) div 2;
          DrawCenteredStringAtPointVertically(textloc,title,GState.FGRasterOp,BothPlanes);
        end;
      end;
    end;
  end; {case tl of}
end;

procedure PlaceTitledBox(var bx : TitledBoxRec; sbox : RectRec; bt : string; tl : Edge ; tp : TextPos);
begin
  with bx do
  begin
    OutLine := sbox;
    FillRegion(OutLine,GState.BGPat,RasterOp_Rep,BothPlanes);
    TitleText := bt;
    TitleLocation :=tl;
    TextPosition :=tp;
    case tl of
      Top :
      begin
        SplitRectY(Outline,TitleBox,DataBox,Gstate.GraphicFontInfo.CharHeight + 2);
      end;
      Bottom :
      begin
        SplitRectY(Outline,DataBox,TitleBox,(RectYSize(OutLine)-(Gstate.GraphicFontInfo.CharHeight + 3)));
      end;
      Left :
      begin
        SplitRectX(Outline,TitleBox,DataBox,(Gstate.GraphicFontInfo.CharWidth + 2));
      end;
      Right :
      begin
        SplitRectX(Outline,DataBox,TitleBox,(RectXSize(OutLine)-(Gstate.GraphicFontInfo.CharWidth + 3)));
      end;
    end;
    DrawRectangle(TitleBox,GState.FGColor,GState.FGRasterOp,BothPlanes);
    DrawRectangle(DataBox,GState.FGColor,GState.FGRasterOp,BothPlanes);
    if (tl = Top) or (tl = Bottom) then
    begin
      case tp of
        LeftJustify   : DrawLeftJustifiedStringInBox(TitleBox,bt,2,GState.FGRasterOp,BothPlanes);
        CenterJustify : DrawCenteredStringInBox(TitleBox,bt,GState.FGRasterOp,BothPlanes);
      end;
    end;
    if (tl = Left) or (tl = Right) then
    begin
      case tp of
        TopJustify    : DrawTopJustifiedStringInBoxVertically(TitleBox,bt,2,GState.FGRasterOp,BothPlanes);
        CenterJustify : DrawCenteredStringInBoxVertically(TitleBox,bt,GState.FGRasterOp,BothPlanes);
      end;
    end;
  end;
end;

function CreateTitledBox(sbox : RectRec; bt : string; tl : Edge ; tp : TextPos) : TitledBoxPtr;
var bx : TitledBoxPtr;
begin
  GetMem(bx,SizeOf(TitledBoxRec));
  with bx^ do
  begin
    BufSize := GetRegionBuffSize(sbox);
    GetMem(BufPtr,BufSize);
    SaveRegion(sbox,BufPtr);
    PlaceTitledBox(bx^,sbox,bt,tl,tp);
    Seal := ValidSeal;
    CreateTitledBox := bx;
  end;
end;

procedure HighLightTitledBox(bx : TitledBoxPtr);
var
hrect : RectRec;
begin
  with bx^ do
  begin
    if Seal = ValidSeal then
    begin
      InsetRect(TitleBox,hrect,-1,-1);
      FillRegion(hrect,Gstate.FGPat,RasterOp_Xor,BothPlanes);
    end;
  end;
end;

procedure GetTitleBox(bx :TitledBoxPtr;var trect :RectRec);
begin
  with bx^ do
  begin
    if Seal = ValidSeal then
    begin
      trect := TitleBox;
    end
    else SystemErrorNotifier('TitldBx:','No-init');
  end;
end;

procedure GetDataBox(bx :TitledBoxPtr;var drect :RectRec);
begin
  with bx^ do
  begin
    if Seal = ValidSeal then
    begin
      drect := DataBox;
    end
    else SystemErrorNotifier('TitldBx:','No-init');
  end;
end;

procedure TitledBoxDestroy(bx : TitledBoxPtr);
begin
  with bx^ do
  begin
    if Seal = ValidSeal then
    begin
      Seal := InvalidSeal;
      RestoreRegion(BufPtr);
      FreeMem(BufPtr,BufSize);
      FreeMem(bx,SizeOf(TitledBoxRec));
    end
    else SystemErrorNotifier('TitldBx:','No-init');
  end;
end;

procedure DoubleBorderTTYWinPlace(thewin : RectRec; blink : boolean);
{Part of TTYWinCreate }
var
smallrect : rectRec;
pm1,pm2 : byte;
begin
  FillRegion(thewin,GState.BGPat,RasterOp_Rep,BothPlanes);
  if blink = true then pm1 := Plane1 else pm1 := BothPlanes;
  DrawRectangle(thewin,GState.FGColor,GState.FGRasterOp,pm1);
  InsetRect(thewin,smallrect,-1,-1);
  if blink then DrawRectangle(smallrect,GState.FGColor,GState.FGRasterOp,Plane2);
  InsetRect(thewin,smallrect,-2,-2);
  DrawRectangle(smallrect,GState.FGColor,GState.FGRasterOp,pm1);
  InsetRect(thewin,smallrect,-4,-4);
  SetClippedTTYWin(smallrect);
end;

procedure SingleBorderTTYWinPlace(thewin : RectRec; blink : boolean);
{ Part of TTY WinCreate }
var
smallrect : rectRec;
pm1,pm2 : byte;
begin
  FillRegion(thewin,GState.BGPat,RasterOp_Rep,BothPlanes);
  if blink = true then pm1 := Plane1 else pm1 := BothPlanes;
  DrawRectangle(thewin,GState.FGColor,GState.FGRasterOp,pm1);
  InsetRect(thewin,smallrect,-2,-2);
  SetClippedTTYWin(smallrect);
end;

procedure NoBorderTTYWinPlace(thewin : RectRec; blink : boolean);
{ Part of TTY WinCreate }
var
smallrect : rectRec;
pm1,pm2 : byte;
begin
  FillRegion(thewin,GState.BGPat,RasterOp_Rep,BothPlanes);
  InsetRect(thewin,smallrect,-1,-1);
  SetClippedTTYWin(smallrect);
end;

function TTYWinCreate(thewin : RectRec; blink : boolean;bw : byte) : TextWindowPtr;
{ Place a tty window on the screen with save under }
{ if blink is true, border blinks (if 2planes)}
{ bw is border width, 0,1, or 2 }
var
tw : TextWindowPtr;
begin
  GetMem(tw,SizeOf(TextWindowRec));
  with tw^ do
  begin
    if (Gstate.TopWindowPtr^.HasFocus and (Gstate.TopWindowPtr^.Seal = ValidSeal)) then
    begin
      WeStoleFocus := true;
      StolenFrom := GState.TopWindowPtr;
      GetTTYState(StolenFrom^.TTYStateInfo);
    end
    else WeStoleFocus := false;
    Outline := thewin;
    SaveUnderBufSize := GetRegionBuffSize(OutLine);
    GetMem(SaveUnderBufPtr,SaveUnderBufSize);
    SaveRegion(OutLine,SaveUnderBufPtr);
    SaveUnderValid := true;
    SaveSelfValid := false;
    BlinkF := blink;
    case bw of
      0: NoBorderTTYWinPlace(thewin,blink);
      1: SingleBorderTTYWinPlace(thewin,blink);
      2: DoubleBorderTTYWinPlace(thewin,blink);
    end;
    GetTTYState(TTYStateInfo);
    HasFocus := true;
    WakeCursor;
    Seal :=ValidSeal;
    Gstate.TopWindowPtr := tw;
    TTYWinCreate := tw;
  end;
end;

procedure TTYWinDestroy(tw :TextWindowPtr);
begin
  with tw^ do
  begin
    if Seal = ValidSeal then
    begin
      Seal := InvalidSeal;
      if SaveUnderValid then
      begin
        RestoreRegion(SaveUnderBufPtr);
        FreeMem(SaveUnderBufPtr,SaveUnderBufSize);
      end;
      if HasFocus then
      begin
        SleepCursor;
        HasFocus := false;
        if WeStoleFocus then
        begin
          SetTTYState(StolenFrom^.TTYStateInfo,false);
          WakeCursor;
        end;
      end;
      FreeMem(tw,SizeOf(TextWindowRec));
    end
    else SystemErrorNotifier('TTYWin:','No-init');
  end;
end;

procedure TTYWinSleep(tw :TextWindowPtr);
begin
  with tw^ do
  begin
    if Seal = ValidSeal then
    begin
      if HasFocus then
      begin
        GetTTYState(TTYStateInfo);
        SleepCursor;
        HasFocus := false;
        if WeStoleFocus then
        begin
          SetTTYState(StolenFrom^.TTYStateInfo,false);
          WakeCursor;
        end;
      end;
    end
    else SystemErrorNotifier('TTYWin:','No-init');
  end;
end;

procedure TTYWinWake(tw :TextWindowPtr);
begin
  with tw^ do
  begin
    if Seal = ValidSeal then
    begin
      if (Gstate.TopWindowPtr^.HasFocus and (Gstate.TopWindowPtr^.Seal = ValidSeal)) then
      begin
        StolenFrom := GState.TopWindowPtr;
        { we are going to steal focus, so we need }
        { to save previous windows text state... }
        GetTTYState(StolenFrom^.TTYStateInfo);
        WeStoleFocus := true;
      end
      else WeStoleFocus := false;
      { restore the ttystate but don't clear the screen }
      SetTTYState(TTYStateInfo,false);
      HasFocus := true;
      Gstate.TopWindowPtr := tw;
      WakeCursor;
    end
    else SystemErrorNotifier('TTYWin:','No-init');
  end;
end;

procedure PopUpReadLn(x,y: word;cols,rows : byte;message : string;var answer : string; blink : boolean);
var
tw : TextWindowPtr;
ourwin : RectRec;
winht,winwd,bs : word;
p: pointer;
begin
  winwd := cols * GState.CurTTYState.FontInfo.CharWidth;
  winht := rows * GState.CurTTYState.FontInfo.CharHeight;
  ourwin.UL.X := x;
  ourwin.UL.Y := y;
  ourwin.LR.X := x + winwd + 8;
  ourwin.LR.Y := y + winht + 8;
  tw := TTYWinCreate(ourwin,blink,2);
  write(message);
  readln(answer);
  TTYWinDestroy(tw);
end;

procedure PopUpWaitForChar(x,y: word;cols,rows : byte;message : string; var sc : word; blink : boolean);
var
tw : TextWindowPtr;
ourwin : RectRec;
winht,winwd,bs : word;
begin
  winwd := cols * GState.CurTTYState.FontInfo.CharWidth;
  winht := rows * GState.CurTTYState.FontInfo.CharHeight;
  ourwin.UL.X := x;
  ourwin.UL.Y := y;
  ourwin.LR.X := x + winwd + 8;
  ourwin.LR.Y := y + winht + 8;
  tw := TTYWinCreate(ourwin,blink,2);
  write(message);
  while not QKeyPressed do;
  sc := QkeyRead;
  TTYWinDestroy(tw);
end;
